<?php

namespace App\Http\Controllers\Business;

use App\Models\Business;
use App\Models\Team;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class TeamController extends Controller
{
    /**
     * Display a listing of business teams.
     */
    public function index(Request $request)
    {
        // Check if user is business owner or has permission
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_team')) {
            abort(403, 'Unauthorized');
        }
//        get business from auth
        $business = auth()->user()->business;;

        $query = $business->teams();

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Search
        if ($request->filled('search')) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        $teams = $query->paginate(15);

        return view('business.teams.index', compact('business', 'teams'));
    }

    /**
     * Show the form for creating a new team.
     */
    public function create()
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can create teams');
        }

        // Get business from authenticated user
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'No business found. Please create one first.');
        }

        // Get only active staff members who have accepted invitations (have users)
        $staff = $business->staff()
            ->whereNotNull('user_id')
            ->where('status', 'active')
            ->with('user')
            ->get();

        return view('business.teams.create', compact('business', 'staff'));
    }

    /**
     * Store a newly created team in storage.
     */
    public function store(Request $request)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can create teams');
        }

        // Get business from authenticated user
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'No business found. Please create one first.');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'status' => 'required|in:active,inactive,archived',
            'members' => 'nullable|array',
            'members.*' => 'integer|exists:users,id',
        ]);

        $team = $business->teams()->create([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'status' => $validated['status'],
            'business_id' => $business->id,  // Explicitly set business_id
        ]);

        // Add members if provided
        if (!empty($validated['members'])) {
            $managerRole = Role::where('name', 'business_manager')->first();
            foreach ($validated['members'] as $userId) {
                $user = User::find($userId);
                $team->members()->attach($userId, [
                    'role_id' => $managerRole?->id,
                    'status' => 'active',
                    'joined_at' => now(),
                ]);

                // Send notification to team member
                $channels = config_value('notifications.channels.team_added', ['database']);
                notify($user, 'team_added', [
                    'team_name' => $team->name,
                    'business_name' => $business->business_name,
                    'action_url' => route('business.teams.show', [$business->id, $team->id]),
                ], $channels);
            }
            $team->update(['member_count' => count($validated['members'])]);
        }

        return redirect()->route('business.teams.show', [$business->id, $team->id])
            ->with('success', 'Team created successfully.');
    }

    /**
     * Display the specified team.
     */
    public function show(Business $business, Team $team)
    {
        // Check authorization
        if (!auth()->user()->isBusinessOwner() && !auth()->user()->hasTeamPermission('manage_team')) {
            abort(403, 'Unauthorized');
        }

        abort_if($team->business_id !== $business->id, 404);

        $team->load('members', 'chats');

        return view('business.teams.show', compact('business', 'team'));
    }

    /**
     * Show the form for editing the team.
     */
    public function edit(Business $business, Team $team)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can edit teams');
        }

        abort_if($team->business_id !== $business->id, 404);

        return view('business.teams.edit', compact('business', 'team'));
    }

    /**
     * Update the specified team in storage.
     */
    public function update(Request $request, Business $business, Team $team)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can update teams');
        }

        abort_if($team->business_id !== $business->id, 404);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:500',
            'status' => 'required|in:active,inactive,archived',
        ]);

        $team->update($validated);

        return redirect()->route('business.teams.show', [$business, $team])
            ->with('success', 'Team updated successfully.');
    }

    /**
     * Remove the specified team from storage.
     */
    public function destroy(Business $business, Team $team)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can delete teams');
        }

        abort_if($team->business_id !== $business->id, 404);

        $team->delete();

        return redirect()->route('business.teams.index', $business)
            ->with('success', 'Team deleted successfully.');
    }

    /**
     * Add member to team.
     */
    public function addMember(Request $request, Business $business, Team $team)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can add team members');
        }

        abort_if($team->business_id !== $business->id, 404);

        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'role_id' => 'nullable|exists:roles,id',
        ]);

        $user = User::findOrFail($validated['user_id']);
        $role = $validated['role_id'] ? Role::findOrFail($validated['role_id']) : null;

        $team->addMember($user, $role, [], 'active');

        return redirect()->route('business.teams.show', [$business, $team])
            ->with('success', 'Member added successfully.');
    }

    /**
     * Remove member from team.
     */
    public function removeMember(Business $business, Team $team, User $user)
    {
        // Check if user is business owner
        if (!auth()->user()->isBusinessOwner()) {
            abort(403, 'Only business owners can remove team members');
        }

        abort_if($team->business_id !== $business->id, 404);

        $team->removeMember($user);

        return redirect()->route('business.teams.show', [$business, $team])
            ->with('success', 'Member removed successfully.');
    }
}

