<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use Illuminate\Http\Request;

class TicketController extends Controller
{
    /**
     * Display tickets list
     */
    public function index()
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $tickets = SupportTicket::where('business_id', $business->id)
            ->withCount('replies')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $stats = [
            'total' => SupportTicket::where('business_id', $business->id)->count(),
            'open' => SupportTicket::where('business_id', $business->id)->where('status', 'open')->count(),
            'pending' => SupportTicket::where('business_id', $business->id)->where('status', 'pending')->count(),
            'resolved' => SupportTicket::where('business_id', $business->id)->where('status', 'resolved')->count(),
        ];

        return view('business.support.tickets', compact('tickets', 'stats'));
    }

    /**
     * Show ticket creation form
     */
    public function create()
    {
        return view('business.support.create');
    }

    /**
     * Store new ticket
     */
    public function store(Request $request)
    {

        $validated = $request->validate([
            'subject' => 'required|string|max:255',
            'category' => 'required|in:technical,billing,general,feature_request',
            'priority' => 'required|in:low,medium,high,urgent',
            'description' => 'required|string',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:5120'
        ]);

        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $attachments = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('ticket-attachments', 'public');
                $attachments[] = $path;
            }
        }

        $ticket = SupportTicket::create([
            'business_id' => $business->id,
            'user_id' => auth()->id(),
            'ticket_number' => 'TKT-' . strtoupper(uniqid()),
            'subject' => $validated['subject'],
            'category' => $validated['category'],
            'priority' => $validated['priority'],
            'description' => $validated['description'],
            'attachments' => json_encode($attachments),
            'status' => 'open',
        ]);

        return redirect()
            ->route('business.support.show', $ticket->id)
            ->with('success', 'Ticket created successfully. Our team will respond soon.');
    }

    /**
     * Show ticket details
     */
    public function show($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $ticket = SupportTicket::where('business_id', $business->id)
            ->with(['replies.user'])
            ->findOrFail($id);

        return view('business.support.show', compact('ticket'));
    }

    /**
     * Reply to ticket
     */
    public function reply(Request $request, $id)
    {
        $validated = $request->validate([
            'message' => 'required|string',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:5120'
        ]);

        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $ticket = SupportTicket::where('business_id', $business->id)
            ->findOrFail($id);

        $attachments = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('ticket-attachments', 'public');
                $attachments[] = $path;
            }
        }

        SupportTicketReply::create([
            'support_ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'message' => $validated['message'],
            'attachments' => !empty($attachments) ? json_encode($attachments) : null,
            'is_staff_reply' => false
        ]);

        $ticket->update([
            'status' => 'pending',
            'updated_at' => now()
        ]);

        return back()->with('success', 'Reply sent successfully.');
    }

    /**
     * Close ticket
     */
    public function close($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $ticket = SupportTicket::where('business_id', $business->id)
            ->findOrFail($id);

        $ticket->update([
            'status' => 'closed',
            'closed_at' => now()
        ]);

        return back()->with('success', 'Ticket closed successfully.');
    }

    /**
     * Reopen ticket
     */
    public function reopen($id)
    {
        $user = auth()->user();
        $business = $user->businesses()->first();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please register your business first');
        }

        $ticket = SupportTicket::where('business_id', $business->id)
            ->findOrFail($id);

        $ticket->update([
            'status' => 'open',
            'closed_at' => null
        ]);

        return back()->with('success', 'Ticket reopened successfully.');
    }
}
