<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Venue;
use App\Models\Category;
use App\Traits\GetUserBusiness;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class VenueController extends Controller
{
    use GetUserBusiness;
    /**
     * Display venue listing
     */
    public function index()
    {
        $business = $this->getUserBusiness();

        if (!$business) {
            return redirect()->route('business.register')
                ->with('error', 'Please complete business registration first.');
        }

        $venues = $business->venues()
            ->withCount(['offers', 'reviews'])
            ->orderBy('created_at', 'desc')
            ->get();

        $stats = [
            'total' => $venues->count(),
            'active' => $venues->where('status', 'active')->count(),
            'inactive' => $venues->where('status', 'inactive')->count(),
            'total_reviews' => $venues->sum('reviews_count'),
        ];

        return view('business.venues.index', compact('venues', 'stats'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        if (auth()->user()->hasRole('business_manager')) {

            $business = auth()->user()->getOwnerBusinessForStaff;
        } else {
            $business = auth()->user()->business;
        }

        if (!$business->canCreateVenues()) {
            $message = 'You need an active subscription and approved payment to create venues.';
            if (!$business->isVerified()) {
                $message = 'Your business must be verified first.';
            } elseif (!$business->hasApprovedPayment()) {
                $message = 'Please complete payment before creating venues.';
            } elseif (!$business->hasActiveSubscription()) {
                $message = 'You need an active subscription to create venues.';
            }
            return redirect()->route('business.subscription.index')
                ->with('error', $message);
        }

        $categories = Category::where('is_active', true)->get();
        return view('business.venues.create', compact('categories'));
    }

    /**
     * Store new venue
     */
    public function store(Request $request)
    {
        if (auth()->user()->hasRole('business_manager')) {

            $business = auth()->user()->getOwnerBusinessForStaff;
        } else {
            $business = auth()->user()->business;
        }

        // Verify business can create venues
        if (!$business->canCreateVenues()) {
            return back()->withErrors([
                'permission' => 'You need an active subscription and approved payment to create venues.'
            ])->withInput();
        }

        // Check subscription limits
        $plan = $business->currentSubscription?->plan;
        if ($plan && $plan->max_venues) {
            $venueCount = $business->venues()->count();
            if ($venueCount >= $plan->max_venues) {
                return back()->withErrors([
                    'limit' => 'You have reached your venue limit. Please upgrade your plan.'
                ])->withInput();
            }
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'address' => 'required|string',
            'address_ar' => 'nullable|string',
            'city' => 'required|string|max:100',
            'area' => 'required|string|max:100',
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'working_hours' => 'nullable|array',
            'amenities' => 'nullable|array',
            'images' => 'nullable|array',
            'images.*' => 'image|mimes:jpeg,png,jpg|max:5120',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120',
            'redeem_code'=> 'required|string|unique:venues,redeem_code',
        ]);

        // Generate slug
        $validated['slug'] = Str::slug($validated['name']) . '-' . Str::random(6);
        $validated['business_id'] = $business->id;

        // Convert arrays to JSON
        $validated['working_hours'] = json_encode($validated['working_hours'] ?? []);
        $validated['amenities'] = json_encode($validated['amenities'] ?? []);

        // Handle featured image upload
        if ($request->hasFile('featured_image')) {
            $path = $request->file('featured_image')->store('venues/featured', 'public');
            $validated['featured_image'] = $path;
        }

        // Handle multiple images upload
        $imagePaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('venues/gallery', 'public');
                $imagePaths[] = $path;
            }
        }
        $validated['images'] = json_encode($imagePaths);
        $validated['status'] = 'pending';
        $venue = Venue::create($validated);

        // Send notification to business owner about new venue
        $channels = config_value('notifications.channels.venue_created', ['database']);
        notify($business->owner, 'venue_created', [
            'venue_name' => $venue->name,
            'city' => $venue->city,
            'category' => $venue->category->name,
            'action_url' => route('business.venues.show', $venue),
        ], $channels);

        return redirect()->route('business.venues.index')
            ->with('success', 'Venue created successfully!');
    }

    /**
     * Show venue details
     */
    public function show($id)
    {
        $business = auth()->user()->business;
        $venue = Venue::where('business_id', $business->id)
            ->with(['category', 'offers', 'reviews'])
            ->withCount(['offers', 'reviews', 'redemptions'])
            ->findOrFail($id);

        $stats = [
            'total_offers' => $venue->offers_count,
            'active_offers' => $venue->offers()->where('status', 'active')->count(),
            'total_redemptions' => $venue->redemptions_count,
            'total_reviews' => $venue->reviews_count,
            'average_rating' => $venue->rating,
            'total_views' => $venue->view_count,
        ];

        $recentRedemptions = $venue->redemptions()
            ->with('user')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();

        return view('business.venues.show', compact('venue', 'stats', 'recentRedemptions'));
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $business = auth()->user()->business;
        $venue = Venue::where('business_id', $business->id)
            ->findOrFail($id);

        $categories = Category::where('is_active', true)->get();

        return view('business.venues.edit', compact('venue', 'categories'));
    }

    /**
     * Update venue
     */
    public function update(Request $request, $id)
    {
        $business = auth()->user()->business;
        $venue = Venue::where('business_id', $business->id)
            ->findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'address' => 'required|string',
            'address_ar' => 'nullable|string',
            'city' => 'required|string|max:100',
            'area' => 'required|string|max:100',
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'working_hours' => 'nullable|array',
            'amenities' => 'nullable|array',
            'status' => 'required|in:active,inactive,under_maintenance',
            'featured_image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120',
            'new_images' => 'nullable|array',
            'new_images.*' => 'image|mimes:jpeg,png,jpg|max:5120',
            'remove_images' => 'nullable|array',
            'redeem_code'=> 'required|string|unique:venues,redeem_code,'.$venue->id,
        ]);

        // Convert arrays to JSON
        $validated['working_hours'] = json_encode($validated['working_hours'] ?? []);
        $validated['amenities'] = json_encode($validated['amenities'] ?? []);

        // Handle featured image upload
        if ($request->hasFile('featured_image')) {
            // Delete old image
            if ($venue->featured_image) {
                Storage::disk('public')->delete($venue->featured_image);
            }
            $path = $request->file('featured_image')->store('venues/featured', 'public');
            $validated['featured_image'] = $path;
        }

        // Handle gallery images
        $existingImages = json_decode($venue->images, true) ?? [];

        // Remove selected images
        if ($request->has('remove_images')) {
            foreach ($request->remove_images as $imageToRemove) {
                if (in_array($imageToRemove, $existingImages)) {
                    Storage::disk('public')->delete($imageToRemove);
                    $existingImages = array_diff($existingImages, [$imageToRemove]);
                }
            }
        }

        // Add new images
        if ($request->hasFile('new_images')) {
            foreach ($request->file('new_images') as $image) {
                $path = $image->store('venues/gallery', 'public');
                $existingImages[] = $path;
            }
        }

        $validated['images'] = json_encode(array_values($existingImages));
        $validated['status'] = 'pending';
        $venue->update($validated);

        return redirect()->route('business.venues.index')
            ->with('success', 'Venue updated successfully!');
    }

    /**
     * Delete venue
     */
    public function destroy($id)
    {
        $business = auth()->user()->business;
        $venue = Venue::where('business_id', $business->id)
            ->findOrFail($id);

        // Delete images
        if ($venue->featured_image) {
            Storage::disk('public')->delete($venue->featured_image);
        }

        if ($venue->images) {
            $images = json_decode($venue->images, true);
            foreach ($images as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $venue->delete();

        return response()->json([
            'success' => true,
            'message' => 'Venue deleted successfully'
        ]);
    }

    /**
     * Toggle venue status
     */
    public function toggleStatus($id)
    {
        $business = auth()->user()->business;

        $venue = Venue::where('business_id', $business->id)
            ->findOrFail($id);

        // ❌ Do NOT allow toggle if status = pending
        if ($venue->status === 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'This venue is pending approval and cannot be updated right now.',
                'status' => $venue->status
            ], 403);
        }

        // ✅ Toggle only active/inactive
        $venue->update([
            'status' => $venue->status === 'active' ? 'inactive' : 'active'
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Venue status updated',
            'status' => $venue->status
        ]);
    }

}
