<?php

namespace App\Http\Controllers\Business;

use App\Http\Controllers\Controller;
use App\Models\Business;
use Illuminate\Http\Request;

class VerificationController extends Controller
{
    /**
     * Show business verification status with rejection checklist
     */
    public function status()
    {
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.dashboard')
                ->with('error', 'No business associated with this account');
        }

        return view('business.verification.status', compact('business'));
    }

    /**
     * Edit business info for resubmission (form)
     */
    public function edit()
    {
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.dashboard')
                ->with('error', 'No business associated with this account');
        }

        // Check if business is rejected and has resubmission attempts left
        if ($business->status !== 'rejected') {
            return redirect()->route('business.dashboard')
                ->with('info', 'Your business is not pending resubmission');
        }

        // Limit resubmission attempts to 3
        if (($business->verification_attempts ?? 0) >= 3) {
            return redirect()->route('business.dashboard')
                ->with('error', 'You have exceeded maximum resubmission attempts. Please contact support.');
        }

        return view('business.verification.edit', compact('business'));
    }

    /**
     * Resubmit business for verification
     */
    public function resubmit(Request $request)
    {
        $business = auth()->user()->business;

        if (!$business) {
            return redirect()->route('business.dashboard')
                ->with('error', 'No business associated with this account');
        }

        // Check if business is rejected
        if ($business->status !== 'rejected') {
            return redirect()->route('business.dashboard')
                ->with('error', 'Your business is not pending resubmission');
        }

        // Check resubmission limit
        if (($business->verification_attempts ?? 0) >= 3) {
            return redirect()->route('business.dashboard')
                ->with('error', 'You have exceeded maximum resubmission attempts');
        }

        // Validate updated information
        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'trade_license_number' => 'required|string|max:100',
            'trade_license_document' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'email' => 'required|email',
            'phone' => 'required|string|max:20',
            'website' => 'nullable|url',
            'description' => 'required|string|min:50|max:1000',
            'logo' => 'nullable|image|max:2048',
        ]);

        // Handle file uploads
        if ($request->hasFile('trade_license_document')) {
            $path = $request->file('trade_license_document')->store('business/licenses', 'public');
            $validated['trade_license_document'] = $path;
        }

        if ($request->hasFile('logo')) {
            $path = $request->file('logo')->store('business/logos', 'public');
            $validated['logo'] = $path;
        }

        // Update business and reset status to pending
        $business->update(array_merge($validated, [
            'status' => 'pending',
            'rejection_checklist' => null,
            'rejection_reason' => null,
        ]));

        // Send notification to admin users (those with 'admin' role)
        $adminUsers = \App\Models\User::role('admin')->get();
        foreach ($adminUsers as $admin) {
            notify($admin, 'business_resubmitted', [
                'business_name' => $business->business_name,
                'owner_name' => $business->owner->full_name,
                'attempt_number' => ($business->verification_attempts ?? 0) + 1,
                'action_url' => route('admin.businesses.show', $business),
            ], ['database']);
        }

        return redirect()->route('business.dashboard')
            ->with('success', 'Business information updated and resubmitted for verification. An admin will review your submission shortly.');
    }
}

