<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Services\Payment\StripeService;
use Illuminate\Http\Request;
use App\Helpers\ConfigHelper;

class StripeController extends Controller
{
    protected $stripeService;

    public function __construct(StripeService $stripeService)
    {
        $this->stripeService = $stripeService;
    }

    /**
     * Create payment intent
     */
    public function createIntent(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        if ($payment->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Payment is not in pending status'
            ], 400);
        }

        try {
            $intent = $this->stripeService->createPaymentIntent(
                $payment->amount,
                $payment->currency,
                [
                    'payment_id' => $payment->id,
                    'user_id' => $payment->user_id,
                    'type' => $payment->type
                ]
            );

            $payment->update([
                'transaction_id' => $intent->id
            ]);

            return response()->json([
                'success' => true,
                'client_secret' => $intent->client_secret,
                'payment_intent_id' => $intent->id
            ]);
        } catch (\Exception $e) {
            Log::error('Stripe payment intent creation failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment intent'
            ], 500);
        }
    }

    /**
     * Process payment
     */
    public function process(Request $request)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
            'payment_id' => 'required|exists:payments,id'
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        try {
            $intent = $this->stripeService->retrievePaymentIntent($request->payment_intent_id);

            if ($intent->status === 'succeeded') {
                $payment->update([
                    'status' => 'completed',
                    'transaction_id' => $intent->id,
                    'payment_details' => json_encode([
                        'stripe_payment_intent' => $intent->id,
                        'amount_received' => $intent->amount_received / 100,
                        'payment_method' => $intent->payment_method,
                    ])
                ]);

                // Activate subscription if payment is for subscription
                if ($payment->subscription_id) {
                    $subscription = $payment->subscription;
                    $subscription->update(['status' => 'active']);

                    // Update business profile if this is a business subscription
                    if ($payment->business_id && $subscription->plan) {
                        $business = \App\Models\Business::find($payment->business_id);

                        if ($business) {
                            $business->update([
                                'subscription_plan_id' => $subscription->subscription_plan_id,
                                'subscription_status' => 'active',
                                'subscription_start_date' => $subscription->start_date,
                                'subscription_end_date' => $subscription->end_date,
                            ]);

                            Log::info('Business subscription updated', [
                                'business_id' => $business->id,
                                'plan_id' => $subscription->subscription_plan_id,
                                'subscription_id' => $subscription->id
                            ]);
                        }
                    }
                }

                return response()->json([
                    'success' => true,
                    'message' => 'Payment processed successfully',
                    'payment' => $payment
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'Payment not successful',
                'status' => $intent->status
            ], 400);

        } catch (\Exception $e) {
            Log::error('Stripe payment processing failed: ' . $e->getMessage());

            $payment->update(['status' => 'failed']);

            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed'
            ], 500);
        }
    }

    /**
     * Handle Stripe webhook
     */
    public function webhook(Request $request)
    {
        $payload = $request->getContent();
        $sig_header = $request->header('Stripe-Signature');
        $endpoint_secret = config('services.stripe.webhook_secret');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sig_header,
                $endpoint_secret
            );
        } catch (\Exception $e) {
            Log::error('Stripe webhook error: ' . $e->getMessage());
            return response()->json(['error' => 'Webhook error'], 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object;
                $this->handlePaymentSuccess($paymentIntent);
                break;

            case 'payment_intent.payment_failed':
                $paymentIntent = $event->data->object;
                $this->handlePaymentFailure($paymentIntent);
                break;

            case 'charge.refunded':
                $charge = $event->data->object;
                $this->handleRefund($charge);
                break;

            default:
                Log::info('Unhandled Stripe webhook event: ' . $event->type);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Handle successful payment
     */
    protected function handlePaymentSuccess($paymentIntent)
    {
        $payment = Payment::where('transaction_id', $paymentIntent->id)->first();

        if ($payment && $payment->status === 'pending') {
            $payment->update([
                'status' => 'completed',
                'payment_details' => json_encode($paymentIntent)
            ]);

            if ($payment->subscription_id) {
                $payment->subscription->update(['status' => 'active']);
            }
        }
    }

    /**
     * Handle failed payment
     */
    protected function handlePaymentFailure($paymentIntent)
    {
        $payment = Payment::where('transaction_id', $paymentIntent->id)->first();

        if ($payment) {
            $payment->update([
                'status' => 'failed',
                'payment_details' => json_encode($paymentIntent)
            ]);
        }
    }

    /**
     * Handle refund
     */
    protected function handleRefund($charge)
    {
        $payment = Payment::where('transaction_id', $charge->payment_intent)->first();

        if ($payment) {
            $payment->update([
                'status' => 'refunded',
                'payment_details' => json_encode($charge)
            ]);
        }
    }

    /**
     * Process refund
     */
    public function refund(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'amount' => 'nullable|numeric|min:0',
            'reason' => 'nullable|string'
        ]);

        $payment = Payment::findOrFail($request->payment_id);

        if ($payment->status !== 'completed') {
            return response()->json([
                'success' => false,
                'message' => 'Only completed payments can be refunded'
            ], 400);
        }

        try {
            $refund = $this->stripeService->refundPayment(
                $payment->transaction_id,
                $request->amount ? $request->amount * 100 : null,
                $request->reason
            );

            $payment->update([
                'status' => 'refunded',
                'payment_details' => json_encode(array_merge(
                    json_decode($payment->payment_details, true) ?? [],
                    ['refund' => $refund]
                ))
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Refund processed successfully',
                'refund' => $refund
            ]);

        } catch (\Exception $e) {
            Log::error('Stripe refund failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Refund processing failed'
            ], 500);
        }
    }

    /**
     * Stripe checkout page
     */
    public function stripeCheckout(Request $request)
    {
        $planId = $request->query('plan_id');
        $plan = \App\Models\SubscriptionPlan::findOrFail($planId);

        // Get business context
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        if (!$business) {
            return redirect()->route('business.register')->with('error', 'No business found');
        }

        try {
            // Create subscription record FIRST with all required fields
            $subscription = \App\Models\Subscription::create([
                'user_id' => auth()->id(),
                'business_id' => $business->id,
                'subscription_plan_id' => $plan->id,
                'status' => 'trial',
                'start_date' => now(),
                'end_date' => now()->addDays($plan->duration_days ?? 30),
                'next_billing_date' => now()->addDays($plan->duration_days ?? 30),
                'auto_renew' => true,
            ]);

            // Then create payment record with valid subscription_id
            $payment = \App\Models\Payment::create([
                'user_id' => auth()->id(),
                'business_id' => $business->id,
                'subscription_id' => $subscription->id,
                'amount' => $plan->price,
                'currency' => 'aed',
                'type' => 'subscription',
                'status' => 'pending',
                'payment_method' => 'stripe',
            ]);

            // Create Stripe Checkout session
            $session = \Stripe\Checkout\Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'aed',
                        'product_data' => [
                            'name' => $plan->name,
                            'description' => $plan->description,
                        ],
                        'unit_amount' => (int)($plan->price * 100), // Convert to cents
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => route('payment.success') . '?session_id={CHECKOUT_SESSION_ID}&payment_id=' . $payment->id,
                'cancel_url' => route('business.subscription.index'),
                'customer_email' => auth()->user()->email,
                'client_reference_id' => $payment->id,
                'metadata' => [
                    'payment_id' => $payment->id,
                    'subscription_id' => $subscription->id,
                    'business_id' => $business->id,
                    'plan_id' => $plan->id,
                ]
            ]);

            // Store session ID and Stripe subscription ID in payment
            $payment->update([
                'transaction_id' => $session->id,
            ]);

            // Store Stripe session ID in subscription
            $subscription->update([
                'stripe_subscription_id' => $session->id,
            ]);

            // Render Stripe checkout view with session
            return view('payment.stripe-checkout', compact('plan', 'business', 'payment', 'subscription', 'session'));
        } catch (\Exception $e) {
            \Log::error('Stripe checkout failed', ['error' => $e->getMessage()]);
            return redirect()->back()->with('error', 'Failed to initiate payment: ' . $e->getMessage());
        }
    }

    /**
     * PayPal checkout page
     */
    public function paypalCheckout(Request $request)
    {
        $planId = $request->query('plan_id');
        $plan = \App\Models\SubscriptionPlan::findOrFail($planId);

        // Get business context
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
        if (!$business) {
            return redirect()->route('business.register')->with('error', 'No business found');
        }

        try {
            // Create subscription record FIRST with all required fields
            $subscription = \App\Models\Subscription::create([
                'user_id' => auth()->id(),
                'business_id' => $business->id,
                'subscription_plan_id' => $plan->id,
                'status' => 'trial',
                'start_date' => now(),
                'end_date' => now()->addDays($plan->duration_days ?? 30),
                'next_billing_date' => now()->addDays($plan->duration_days ?? 30),
                'auto_renew' => true,
            ]);

            // Then create payment record with valid subscription_id
            $payment = \App\Models\Payment::create([
                'user_id' => auth()->id(),
                'business_id' => $business->id,
                'subscription_id' => $subscription->id,
                'amount' => $plan->price,
                'currency' => 'aed',
                'type' => 'subscription',
                'status' => 'pending',
                'payment_method' => 'paypal',
            ]);


            // Render PayPal payment form view
            return view('payment.paypal-checkout', compact('plan', 'business', 'payment', 'subscription'));
        } catch (\Exception $e) {
            \Log::error('PayPal checkout failed', ['error' => $e->getMessage()]);
            return redirect()->back()->with('error', 'Failed to initiate payment: ' . $e->getMessage());
        }
    }

    /**
     * Show checkout page for subscription upgrade
     */
    public function checkout(Request $request)
    {
        $planId = $request->query('plan_id');
        $type = $request->query('type', 'business');

        $plan = \App\Models\SubscriptionPlan::findOrFail($planId);

        if ($plan->type !== $type) {
            return redirect()->back()->with('error', 'Invalid plan selected');
        }

        // Get business context
        $business = null;
        if ($type === 'business') {
            $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;
            if (!$business) {
                return redirect()->route('business.register');
            }
        }

        // Get current subscription
        $currentSubscription = $business ? $business->currentBusinessSubscription : null;

        // Load payment configuration using helper
        $stripeEnabled = ConfigHelper::isPaymentMethodEnabled('stripe');
        $paypalEnabled = ConfigHelper::isPaymentMethodEnabled('paypal');

        // If no payment methods available
        if (!$stripeEnabled && !$paypalEnabled) {
            return redirect()->back()->with('error', 'No payment methods are currently available. Please contact support.');
        }

        return view('payment.checkout', compact('plan', 'business', 'currentSubscription', 'type', 'stripeEnabled', 'paypalEnabled'));
    }

    /**
     * Process subscription checkout payment
     */
    public function processCheckout(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|exists:subscription_plans,id',
            'payment_method_id' => 'required|string',
        ]);

        $plan = \App\Models\SubscriptionPlan::findOrFail($request->plan_id);
        $business = auth()->user()->business ?? auth()->user()->businessStaff?->first()?->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'Business not found'
            ], 400);
        }

        try {
            // Create payment intent
            $intent = $this->stripeService->createPaymentIntent(
                $plan->price * 100, // Convert to cents
                'aed',
                [
                    'business_id' => $business->id,
                    'plan_id' => $plan->id,
                    'type' => 'subscription'
                ]
            );

            // Create payment record
            $payment = Payment::create([
                'user_id' => auth()->id(),
                'business_id' => $business->id,
                'amount' => $plan->price,
                'currency' => 'aed',
                'type' => 'subscription',
                'status' => 'pending',
                'transaction_id' => $intent->id,
                'payment_details' => json_encode([
                    'plan_id' => $plan->id,
                    'plan_name' => $plan->name
                ])
            ]);

            // Create subscription record (will be activated after payment)
            $subscription = \App\Models\BusinessSubscription::create([
                'business_id' => $business->id,
                'subscription_plan_id' => $plan->id,
                'status' => 'trial',
                'start_date' => now(),
                'end_date' => now()->addDays($plan->duration_days ?? 30),
                'next_billing_date' => now()->addDays($plan->duration_days ?? 30),
                'auto_renew' => true,
            ]);

            $payment->update(['subscription_id' => $subscription->id]);

            return response()->json([
                'success' => true,
                'client_secret' => $intent->client_secret,
                'payment_intent_id' => $intent->id,
                'subscription_id' => $subscription->id
            ]);

        } catch (\Exception $e) {
            \Log::error('Subscription checkout failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Checkout processing failed'
            ], 500);
        }
    }

    /**
     * Payment success page
     */
    public function success(Request $request)
    {
        $sessionId = $request->query('session_id');
        $paymentId = $request->query('payment_id');

        if (!$sessionId || !$paymentId) {
            return redirect()->route('business.subscription.index')->with('error', 'Invalid payment session');
        }

        try {
            $payment = Payment::findOrFail($paymentId);

            // Retrieve Stripe session to verify payment
            $session = \Stripe\Checkout\Session::retrieve($sessionId);

            if ($session->payment_status === 'paid') {
                // Update payment status
                $payment->update([
                    'status' => 'completed',
                    'payment_details' => json_encode([
                        'session_id' => $session->id,
                        'payment_intent' => $session->payment_intent,
                        'customer_email' => $session->customer_email,
                    ])
                ]);

                // Activate subscription and update business
                if ($payment->subscription_id) {
                    $subscription = $payment->subscription;
                    $subscription->update(['status' => 'active']);

                    // Update business subscription status
                    if ($payment->business_id) {
                        $business = \App\Models\Business::find($payment->business_id);

                        if ($business && $subscription->subscription_plan_id) {
                            // Create or update business_subscriptions record
                            $businessSubscription = \App\Models\BusinessSubscription::updateOrCreate(
                                [
                                    'business_id' => $business->id,
                                    'subscription_plan_id' => $subscription->subscription_plan_id,
                                ],
                                [
                                    'status' => 'active',
                                    'start_date' => $subscription->start_date ?? now(),
                                    'end_date' => $subscription->end_date ?? now()->addDays(30),
                                    'next_billing_date' => $subscription->next_billing_date ?? now()->addDays(30),
                                    'auto_renew' => $subscription->auto_renew ?? true,
                                ]
                            );

                            // Also update business record
                            $plan = \App\Models\SubscriptionPlan::find($subscription->subscription_plan_id);
                            if ($plan) {
                                $business->update([
                                    'subscription_plan_id' => $plan->id,
                                    'subscription_status' => 'active',
                                    'subscription_start_date' => $subscription->start_date ?? now(),
                                    'subscription_end_date' => $subscription->end_date ?? now()->addDays(30),
                                ]);
                            }

                            \Log::info('Business subscription activated after payment', [
                                'payment_id' => $payment->id,
                                'subscription_id' => $subscription->id,
                                'business_id' => $business->id,
                                'business_subscription_id' => $businessSubscription->id,
                                'plan_id' => $subscription->subscription_plan_id,
                                'amount' => $payment->amount,
                            ]);
                        }
                    }

                    // Log success
                    \Log::info('Payment completed successfully', [
                        'payment_id' => $payment->id,
                        'subscription_id' => $subscription->id,
                        'amount' => $payment->amount,
                    ]);
                }

                return view('payment.success', compact('payment'));
            }

            return redirect()->route('business.subscription.index')->with('error', 'Payment was not completed');

        } catch (\Exception $e) {
            \Log::error('Payment success handler failed', ['error' => $e->getMessage()]);
            return redirect()->route('business.subscription.index')->with('error', 'Error processing payment');
        }
    }

    /**
     * PayPal payment completion
     */
    public function paypalComplete(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'subscription_id' => 'required|exists:subscriptions,id',
            'order_id' => 'required|string',
        ]);

        try {
            $payment = Payment::findOrFail($request->payment_id);
            $subscription = \App\Models\Subscription::findOrFail($request->subscription_id);

            // Update payment status
            $payment->update([
                'status' => 'completed',
                'transaction_id' => $request->order_id,
                'payment_details' => json_encode([
                    'order_id' => $request->order_id,
                    'payment_method' => 'paypal',
                ])
            ]);

            // Activate subscription
            $subscription->update(['status' => 'active']);

            // Update business subscription status
            if ($payment->business_id && $subscription->subscription_plan_id) {
                $business = \App\Models\Business::find($payment->business_id);

                if ($business) {
                    // Create or update business_subscriptions record
                    $businessSubscription = \App\Models\BusinessSubscription::updateOrCreate(
                        [
                            'business_id' => $business->id,
                            'subscription_plan_id' => $subscription->subscription_plan_id,
                        ],
                        [
                            'status' => 'active',
                            'start_date' => $subscription->start_date ?? now(),
                            'end_date' => $subscription->end_date ?? now()->addDays(30),
                            'next_billing_date' => $subscription->next_billing_date ?? now()->addDays(30),
                            'auto_renew' => $subscription->auto_renew ?? true,
                        ]
                    );

                    // Also update business record
                    $plan = \App\Models\SubscriptionPlan::find($subscription->subscription_plan_id);
                    if ($plan) {
                        $business->update([
                            'subscription_plan_id' => $plan->id,
                            'subscription_status' => 'active',
                            'subscription_start_date' => $subscription->start_date ?? now(),
                            'subscription_end_date' => $subscription->end_date ?? now()->addDays(30),
                        ]);
                    }

                    \Log::info('Business subscription activated after PayPal payment', [
                        'payment_id' => $payment->id,
                        'subscription_id' => $subscription->id,
                        'business_id' => $business->id,
                        'business_subscription_id' => $businessSubscription->id,
                        'plan_id' => $subscription->subscription_plan_id,
                        'amount' => $payment->amount,
                    ]);
                }
            }

            \Log::info('PayPal payment completed successfully', [
                'payment_id' => $payment->id,
                'subscription_id' => $subscription->id,
                'order_id' => $request->order_id,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment completed successfully',
                'payment_id' => $payment->id,
            ]);

        } catch (\Exception $e) {
            \Log::error('PayPal completion failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to complete payment: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
    {
        $subscriptionId = $request->query('subscription_id');
        $subscription = \App\Models\BusinessSubscription::findOrFail($subscriptionId);

        $business = $subscription->business;

        if ($business->owner_id !== auth()->id()) {
            return redirect()->back()->with('error', 'Unauthorized');
        }

        return view('payment.renew', compact('subscription', 'business'));
    }

    /**
     * Process subscription renewal
     */
    public function processRenew(Request $request)
    {
        $request->validate([
            'subscription_id' => 'required|exists:business_subscriptions,id',
            'payment_method_id' => 'required|string',
        ]);

        $subscription = \App\Models\BusinessSubscription::findOrFail($request->subscription_id);
        $business = $subscription->business;

        if ($business->owner_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            $plan = $subscription->plan;

            // Create payment intent
            $intent = $this->stripeService->createPaymentIntent(
                $plan->price * 100, // Convert to cents
                'aed',
                [
                    'business_id' => $business->id,
                    'subscription_id' => $subscription->id,
                    'type' => 'renewal'
                ]
            );

            // Create payment record
            $payment = Payment::create([
                'business_id' => $business->id,
                'subscription_id' => $subscription->id,
                'amount' => $plan->price,
                'currency' => 'aed',
                'type' => 'renewal',
                'status' => 'pending',
                'transaction_id' => $intent->id,
            ]);

            return response()->json([
                'success' => true,
                'client_secret' => $intent->client_secret,
                'payment_intent_id' => $intent->id
            ]);

        } catch (\Exception $e) {
            Log::error('Subscription renewal failed: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Renewal processing failed'
            ], 500);
        }
    }
}
