<?php

namespace App\Http\Controllers\Payment;

use App\Http\Controllers\Controller;
use App\Services\Payment\StripeService;
use App\Services\Payment\PayPalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookController extends Controller
{
    /**
     * Handle Stripe webhook
     */
    public function stripe(Request $request)
    {
        // Check if Stripe is enabled in configuration
        $stripeEnabled = config_value('payment.stripe.enabled', true);
        if (!$stripeEnabled) {
            Log::warning('Stripe webhook received but Stripe is disabled in configuration');
            return response()->json(['error' => 'Stripe is disabled'], 403);
        }

        $payload = $request->getContent();
        $signature = $request->header('Stripe-Signature');

        try {
            $stripeService = new StripeService();
            $result = $stripeService->handleWebhook($payload, $signature);

            if ($result['success']) {
                // Send payment confirmation notification if payment completed
                if (isset($result['payment']) && $result['payment']->status === 'completed') {
                    $user = $result['payment']->user;
                    $channels = config_value('notifications.channels.payment_confirmed', ['database', 'email']);

                    notify($user, 'payment_confirmed', [
                        'amount' => $result['payment']->amount,
                        'currency' => $result['payment']->currency,
                        'payment_method' => 'Stripe',
                        'transaction_id' => $result['payment']->transaction_id,
                        'date' => now()->format('Y-m-d H:i'),
                        'action_url' => route('user.payments'),
                    ], $channels);
                }

                return response()->json(['status' => 'success'], 200);
            } else {
                Log::error('Stripe webhook failed', ['error' => $result['error']]);
                return response()->json(['error' => $result['error']], 400);
            }
        } catch (\Exception $e) {
            Log::error('Stripe webhook exception', ['message' => $e->getMessage()]);
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Handle PayPal webhook
     */
    public function paypal(Request $request)
    {
        // Check if PayPal is enabled in configuration
        $paypalEnabled = config_value('payment.paypal.enabled', true);
        if (!$paypalEnabled) {
            Log::warning('PayPal webhook received but PayPal is disabled in configuration');
            return response()->json(['error' => 'PayPal is disabled'], 403);
        }

        try {
            $paypalService = new PayPalService();
            $result = $paypalService->handleWebhook($request->all(), $request->headers->all());

            if ($result['success']) {
                // Send payment confirmation notification if payment completed
                if (isset($result['payment']) && $result['payment']->status === 'completed') {
                    $user = $result['payment']->user;
                    $channels = config_value('notifications.channels.payment_confirmed', ['database', 'email']);

                    notify($user, 'payment_confirmed', [
                        'amount' => $result['payment']->amount,
                        'currency' => $result['payment']->currency,
                        'payment_method' => 'PayPal',
                        'transaction_id' => $result['payment']->transaction_id,
                        'date' => now()->format('Y-m-d H:i'),
                        'action_url' => route('user.payments'),
                    ], $channels);
                }

                return response()->json(['status' => 'success'], 200);
            } else {
                Log::error('PayPal webhook failed', ['error' => $result['error']]);
                return response()->json(['error' => $result['error']], 400);
            }
        } catch (\Exception $e) {
            Log::error('PayPal webhook exception', ['message' => $e->getMessage()]);
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
