<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class AdminSecurityMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Check if user is authenticated
        if (!auth()->check()) {
            return redirect()->route('admin.login');
        }

        // Check if user has admin role
        if (!auth()->user()->hasAnyRole(['super_admin', 'admin'])) {
            abort(403, 'Unauthorized access to admin panel');
        }

        // Check if user's IP is allowed (optional - based on configuration)
        if (!$this->isIpAllowed($request)) {
            abort(403, 'Your IP address is not allowed to access the admin panel');
        }

        // Check if user session is still valid
        if ($this->isSessionExpired($request)) {
            auth()->logout();
            return redirect()->route('admin.login')->with('error', 'Your session has expired');
        }

        // Log admin access
        $this->logAdminAccess($request);

        return $next($request);
    }

    /**
     * Check if user's IP is allowed
     */
    private function isIpAllowed(Request $request): bool
    {
        // Get configuration for IP restrictions
        $ipRestrictions = \App\Models\AdminIpRestriction::where('user_id', auth()->id())
            ->where('is_active', true)
            ->get();

        // If no restrictions are set, allow access
        if ($ipRestrictions->isEmpty()) {
            return true;
        }

        $userIp = $request->ip();

        // Check if user's current IP matches any allowed IPs
        foreach ($ipRestrictions as $restriction) {
            if ($this->ipMatches($userIp, $restriction->ip_address)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP matches (supports CIDR notation)
     */
    private function ipMatches(string $ip, string $cidr): bool
    {
        // Simple IP match
        if ($ip === $cidr) {
            return true;
        }

        // CIDR notation support
        if (strpos($cidr, '/') !== false) {
            list($subnet, $bits) = explode('/', $cidr);
            $ip = ip2long($ip);
            $subnet = ip2long($subnet);
            $mask = -1 << (32 - $bits);
            $subnet &= $mask;
            return ($ip & $mask) === $subnet;
        }

        return false;
    }

    /**
     * Check if session is expired
     */
    private function isSessionExpired(Request $request): bool
    {
        // Get session timeout from configuration (default: 1 hour)
        $timeout = config('admin.session_timeout', 3600);

        // Check admin session
        $adminSession = \App\Models\AdminSession::where('user_id', auth()->id())
            ->where('is_active', true)
            ->latest('last_activity_at')
            ->first();

        if (!$adminSession) {
            return false;
        }

        // Check if last activity was too long ago
        if ($adminSession->last_activity_at->addSeconds($timeout)->isPast()) {
            $adminSession->update(['is_active' => false]);
            return true;
        }

        // Update last activity timestamp
        $adminSession->update(['last_activity_at' => now()]);

        return false;
    }

    /**
     * Log admin access
     */
    private function logAdminAccess(Request $request): void
    {
        // Log admin activity for audit trail
        if ($this->shouldLogActivity($request)) {
            \App\Models\AdminActivityLog::create([
                'admin_user_id' => auth()->id(),
                'action' => 'access',
                'entity_type' => 'admin_panel',
                'entity_id' => auth()->id(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'status' => 'success',
                'is_sensitive' => false,
                'description' => 'Admin accessed panel: ' . $request->path(),
            ]);
        }
    }

    /**
     * Check if activity should be logged
     */
    private function shouldLogActivity(Request $request): bool
    {
        // Don't log static file requests or API heartbeats
        $ignorePaths = [
            'admin/dashboard/statistics',
            'admin/analytics/data',
            'admin/dashboard/activity-summary',
        ];

        foreach ($ignorePaths as $ignorePath) {
            if (str_contains($request->path(), $ignorePath)) {
                return false;
            }
        }

        return true;
    }
}

