<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class AdminActivityLog extends Model
{
    use HasFactory;

    protected $table = 'admin_activity_logs';

    protected $fillable = [
        'admin_user_id',
        'action',
        'entity_type',
        'entity_id',
        'entity_name',
        'old_values',
        'new_values',
        'changes',
        'ip_address',
        'user_agent',
        'description',
        'status',
        'requires_approval',
        'is_sensitive',
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
        'changes' => 'array',
        'requires_approval' => 'boolean',
        'is_sensitive' => 'boolean',
    ];

    /**
     * Admin user who performed the action
     */
    public function adminUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'admin_user_id');
    }

    /**
     * Get change summary
     */
    public function getChangeSummary(): string
    {
        if ($this->changes) {
            $changes = collect($this->changes)->map(function ($change, $field) {
                return "{$field}: {$change['from']} → {$change['to']}";
            })->implode(', ');
            return $changes;
        }

        return match ($this->action) {
            'create' => "Created new {$this->entity_type}",
            'delete' => "Deleted {$this->entity_type}",
            'view' => "Viewed {$this->entity_type}",
            'export' => "Exported {$this->entity_type} data",
            default => ucfirst($this->action) . " {$this->entity_type}",
        };
    }

    /**
     * Scope to logs for a specific action
     */
    public function scopeAction($query, string $action)
    {
        return $query->where('action', $action);
    }

    /**
     * Scope to logs for a specific entity type
     */
    public function scopeForEntity($query, string $entityType)
    {
        return $query->where('entity_type', $entityType);
    }

    /**
     * Scope to sensitive operations
     */
    public function scopeSensitive($query)
    {
        return $query->where('is_sensitive', true);
    }

    /**
     * Scope to failed operations
     */
    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    /**
     * Scope to operations by specific admin
     */
    public function scopeByAdmin($query, $userId)
    {
        return $query->where('admin_user_id', $userId);
    }

    /**
     * Scope to recent logs
     */
    public function scopeRecent($query, int $minutes = 60)
    {
        return $query->where('created_at', '>=', now()->subMinutes($minutes));
    }

    /**
     * Create log entry from request
     */
    public static function createFromRequest(
        $adminUserId,
        $action,
        $entityType,
        $entityId = null,
        $entityName = null,
        $oldValues = null,
        $newValues = null,
        $description = null,
        $isSensitive = false
    ): self {
        $changes = null;

        // Calculate changes if old and new values provided
        if ($oldValues && $newValues) {
            $changes = [];
            foreach ($newValues as $key => $newValue) {
                $oldValue = $oldValues[$key] ?? null;
                if ($oldValue !== $newValue) {
                    $changes[$key] = [
                        'from' => $oldValue,
                        'to' => $newValue,
                    ];
                }
            }
        }

        return self::create([
            'admin_user_id' => $adminUserId,
            'action' => $action,
            'entity_type' => $entityType,
            'entity_id' => $entityId,
            'entity_name' => $entityName,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'changes' => $changes,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'description' => $description,
            'status' => 'success',
            'is_sensitive' => $isSensitive,
        ]);
    }
}

