<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class AdminMfaSetting extends Model
{
    use HasFactory;

    protected $table = 'admin_mfa_settings';

    protected $fillable = [
        'user_id',
        'mfa_method',
        'secret_key',
        'is_enabled',
        'enabled_at',
        'backup_codes',
        'verified_at',
        'failed_attempts',
        'locked_until',
    ];

    protected $casts = [
        'is_enabled' => 'boolean',
        'enabled_at' => 'datetime',
        'verified_at' => 'datetime',
        'locked_until' => 'datetime',
        'backup_codes' => 'array',
    ];

    /**
     * User relationship
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if MFA is enabled and verified
     */
    public function isActivelyEnabled(): bool
    {
        return $this->is_enabled && $this->verified_at !== null;
    }

    /**
     * Check if account is locked due to failed attempts
     */
    public function isLocked(): bool
    {
        if ($this->locked_until === null) {
            return false;
        }

        if ($this->locked_until->isPast()) {
            $this->update([
                'locked_until' => null,
                'failed_attempts' => 0,
            ]);
            return false;
        }

        return true;
    }

    /**
     * Increment failed attempts
     */
    public function incrementFailedAttempts(): void
    {
        $this->increment('failed_attempts');

        // Lock after 5 failed attempts for 30 minutes
        if ($this->failed_attempts >= 5) {
            $this->update(['locked_until' => now()->addMinutes(30)]);
        }
    }

    /**
     * Reset failed attempts
     */
    public function resetFailedAttempts(): void
    {
        $this->update([
            'failed_attempts' => 0,
            'locked_until' => null,
        ]);
    }

    /**
     * Generate backup codes
     */
    public function generateBackupCodes(int $count = 10): array
    {
        $codes = [];
        for ($i = 0; $i < $count; $i++) {
            $codes[] = strtoupper(bin2hex(random_bytes(4)));
        }

        $this->update(['backup_codes' => $codes]);

        return $codes;
    }

    /**
     * Use a backup code
     */
    public function useBackupCode(string $code): bool
    {
        $codes = $this->backup_codes ?? [];

        if (in_array($code, $codes)) {
            $codes = array_filter($codes, fn($c) => $c !== $code);
            $this->update(['backup_codes' => $codes]);
            return true;
        }

        return false;
    }

    /**
     * Scope for enabled MFA
     */
    public function scopeEnabled($query)
    {
        return $query->where('is_enabled', true)->whereNotNull('verified_at');
    }

    /**
     * Scope for specific MFA method
     */
    public function scopeMethod($query, string $method)
    {
        return $query->where('mfa_method', $method);
    }
}

