<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Business extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'owner_id',
        'business_name',
        'business_name_ar',
        'trade_license_number',
        'trade_license_document',
        'email',
        'phone',
        'website',
        'description',
        'description_ar',
        'logo',
        'cover_image',
        'status',
        'verification_notes',
        'verified_at',
        'verified_by',
        'rejection_checklist',
        'rejection_reason',
        'verification_attempts',
        'subscription_plan_id',
        'subscription_status',
        'subscription_start_date',
        'subscription_end_date',
        'total_venues',
        'total_staff',
        'total_offers',
        'rating',
        'total_reviews',
    ];

    protected $casts = [
        'verified_at' => 'datetime',
        'subscription_start_date' => 'date',
        'subscription_end_date' => 'date',
        'rating' => 'decimal:2',
        'rejection_checklist' => 'array',
    ];

    public function owner()
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function verifier()
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function subscriptionPlan()
    {
        return $this->belongsTo(SubscriptionPlan::class);
    }

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscription()
    {
        return $this->hasOne(Subscription::class)
            ->where('status', 'active')
            ->where('end_date', '>', now())
            ->latest();
    }

    public function businessSubscriptions()
    {
        return $this->hasMany(BusinessSubscription::class);
    }

    public function currentBusinessSubscription()
    {
        return $this->hasOne(BusinessSubscription::class)
            ->where('status', 'active')
            ->where('end_date', '>=', now())
            ->latest();
    }

    public function businessSubscriptionHistory()
    {
        return $this->hasMany(BusinessSubscription::class);
    }

    public function marketingSubscriptions()
    {
        return $this->hasMany(MarketingSubscription::class);
    }

    public function currentMarketingSubscription()
    {
        return $this->hasOne(MarketingSubscription::class)
            ->where('status', 'active')
            ->where('end_date', '>', now())
            ->latest();
    }

    public function marketingSubscriptionHistory()
    {
        return $this->hasMany(MarketingSubscription::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function followers()
    {
        return $this->hasMany(Follow::class);
    }

    public function venues()
    {
        return $this->hasMany(Venue::class);
    }

    public function offers()
    {
        return $this->hasMany(Offer::class);
    }

    public function campaigns()
    {
        return $this->hasMany(MarketingCampaign::class);
    }

    public function redemptions()
    {
        return $this->hasManyThrough(Redemption::class, Offer::class);
    }

    public function reviews()
    {
        return $this->hasManyThrough(Review::class, Venue::class);
    }

    public function staff()
    {
        return $this->hasMany(BusinessStaff::class);
    }

    public function teams()
    {
        return $this->hasMany(Team::class);
    }

    public function activeStaff()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'active');
    }

    public function pendingInvitations()
    {
        return $this->hasMany(BusinessStaff::class)->where('status', 'pending_invitation');
    }

    public function managers()
    {
        return $this->hasMany(BusinessStaff::class)
            ->where('role', 'manager')
            ->where('status', 'active');
    }

    public function isVerified()
    {
        return $this->status === 'verified';
    }

    public function canAddMoreStaff()
    {
        if (!$this->subscriptionPlan || !$this->subscriptionPlan->max_staff) {
            return true; // No limit
        }

        $currentStaffCount = $this->staff()
            ->where('status', '!=', 'inactive')
            ->count();

        return $currentStaffCount < $this->subscriptionPlan->max_staff;
    }

    public function hasApprovedPayment()
    {
        return $this->payments()
            ->where('status', 'completed')
            ->exists();
    }

    public function canCreateVenues()
    {
        return $this->isVerified() && $this->hasApprovedPayment() && $this->currentBusinessSubscription !== null;
    }

    public function canCreateOffers()
    {
        return $this->isVerified() && $this->hasApprovedPayment() && $this->currentBusinessSubscription !== null;
    }

    /**
     * Check if business can add a new venue
     */
    public function canAddVenue(): bool
    {
        $subscription = $this->currentBusinessSubscription;
        if (!$subscription) return false;

        $plan = $subscription->plan;
        if ($plan->max_venues == -1) return true; // Unlimited

        return $this->venues()->count() < $plan->max_venues;
    }

    /**
     * Check if business can add staff
     */
    public function canAddStaff(): bool
    {
        $subscription = $this->currentBusinessSubscription;
        if (!$subscription) return false;

        $plan = $subscription->plan;
        if ($plan->max_staff == -1) return true; // Unlimited

        return $this->activeStaff()->count() < $plan->max_staff;
    }

    /**
     * Check if business can create campaign this month
     */
    public function canCreateCampaign(): bool
    {
        $subscription = $this->currentBusinessSubscription;
        if (!$subscription) return false;

        $plan = $subscription->plan;
        if ($plan->monthly_campaigns == -1) return true; // Unlimited

        $thisMonth = MarketingCampaign::where('business_id', $this->id)
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();

        return $thisMonth < $plan->monthly_campaigns;
    }

    /**
     * Get remaining campaign slots for this month
     */
    public function getRemainingCampaignSlots(): int
    {
        $subscription = $this->currentBusinessSubscription;
        if (!$subscription) return 0;

        $plan = $subscription->plan;
        if ($plan->monthly_campaigns == -1) return -1; // Unlimited

        $used = MarketingCampaign::where('business_id', $this->id)
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();

        return max(0, $plan->monthly_campaigns - $used);
    }

    /**
     * Get subscription tier
     */
    public function getSubscriptionTier(): ?string
    {
        $subscription = $this->currentBusinessSubscription;
        return $subscription ? $subscription->plan->tier : null;
    }

    /**
     * Check if business is on specific tier
     */
    public function isOnTier($tier): bool
    {
        return $this->getSubscriptionTier() === $tier;
    }

    /**
     * Check if subscription is active
     */
    public function hasActiveSubscription(): bool
    {
        return $this->currentBusinessSubscription !== null;
    }

    /**
     * Get days until subscription expires
     */
    public function getDaysUntilSubscriptionExpires(): int
    {
        $subscription = $this->currentBusinessSubscription;
        return $subscription ? $subscription->daysRemaining() : 0;
    }

    /**
     * Get subscription usage details
     */
    public function getSubscriptionUsage(): array
    {
        $subscription = $this->currentBusinessSubscription;
        if (!$subscription) return [];

        $plan = $subscription->plan;

        return [
            'venues' => [
                'used' => $this->venues()->count(),
                'limit' => $plan->max_venues,
                'unlimited' => $plan->max_venues == -1,
            ],
            'staff' => [
                'used' => $this->activeStaff()->count(),
                'limit' => $plan->max_staff,
                'unlimited' => $plan->max_staff == -1,
            ],
            'offers' => [
                'used' => $this->offers()->count(),
                'limit' => $plan->max_offers,
                'unlimited' => $plan->max_offers == -1,
            ],
            'campaigns_this_month' => [
                'used' => MarketingCampaign::where('business_id', $this->id)
                    ->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year)
                    ->count(),
                'limit' => $plan->monthly_campaigns,
                'unlimited' => $plan->monthly_campaigns == -1,
            ]
        ];
    }
}
