<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BusinessSubscription extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'business_subscriptions';

    protected $fillable = [
        'business_id',
        'subscription_plan_id',
        'status',
        'start_date',
        'end_date',
        'trial_end_date',
        'auto_renew',
        'payment_method',
        'next_billing_date',
        'cancelled_at',
        'cancellation_reason',
        'usage',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'trial_end_date' => 'date',
        'next_billing_date' => 'date',
        'cancelled_at' => 'datetime',
        'auto_renew' => 'boolean',
        'usage' => 'array',
    ];

    /**
     * Business relationship
     */
    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    /**
     * Subscription plan relationship
     */
    public function plan(): BelongsTo
    {
        return $this->belongsTo(SubscriptionPlan::class, 'subscription_plan_id');
    }

    /**
     * Scope for active subscriptions
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
            ->where('end_date', '>=', now());
    }

    /**
     * Scope for trial subscriptions
     */
    public function scopeTrial($query)
    {
        return $query->where('status', 'trial')
            ->where('trial_end_date', '>=', now());
    }

    /**
     * Check if subscription is active
     */
    public function isActive(): bool
    {
        return $this->status === 'active' && $this->end_date >= now();
    }

    /**
     * Check if subscription is in trial period
     */
    public function isInTrial(): bool
    {
        return $this->status === 'trial'
            && $this->trial_end_date
            && $this->trial_end_date >= now();
    }

    /**
     * Check if subscription is expiring soon
     */
    public function isExpiringSoon($days = 7): bool
    {
        return $this->isActive()
            && $this->end_date->lessThanOrEqualTo(now()->addDays($days))
            && $this->end_date->greaterThan(now());
    }

    /**
     * Get days remaining
     */
    public function daysRemaining(): int
    {
        if (!$this->isActive()) {
            return 0;
        }
        return now()->diffInDays($this->end_date);
    }

    /**
     * Get usage for a specific key
     */
    public function getUsage($key, $default = 0)
    {
        $usage = $this->usage ?? [];
        return $usage[$key] ?? $default;
    }

    /**
     * Update usage
     */
    public function updateUsage($key, $value)
    {
        $usage = $this->usage ?? [];
        $usage[$key] = $value;
        $this->update(['usage' => $usage]);
    }

    /**
     * Increment usage
     */
    public function incrementUsage($key, $amount = 1)
    {
        $current = $this->getUsage($key, 0);
        $this->updateUsage($key, $current + $amount);
    }
}

