<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Chat extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'team_id',
        'business_id',
        'name',
        'description',
        'type',
        'created_by',
        'is_active',
        'message_count',
        'last_message_at',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
        'last_message_at' => 'datetime',
    ];

    // =============== Relationships ===============

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function messages(): HasMany
    {
        return $this->hasMany(Message::class)->orderBy('created_at', 'desc');
    }

    public function participants(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'chat_participants')
            ->withPivot('role', 'joined_at', 'last_read_at', 'is_active')
            ->withTimestamps();
    }

    // =============== Scopes ===============

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeForTeam($query, $teamId)
    {
        return $query->where('team_id', $teamId);
    }

    // =============== Methods ===============

    /**
     * Add participants to chat
     */
    public function addParticipants(array $userIds, $role = 'member'): void
    {
        foreach ($userIds as $userId) {
            $this->participants()->attach($userId, [
                'role' => $role,
                'joined_at' => now(),
                'is_active' => true,
            ]);
        }
    }

    /**
     * Remove participant from chat
     */
    public function removeParticipant(User $user): void
    {
        $this->participants()->detach($user->id);
    }

    /**
     * Post a message
     */
    public function postMessage(User $sender, string $content, string $type = 'text', ?array $metadata = null): Message
    {
        $message = $this->messages()->create([
            'sender_id' => $sender->id,
            'content' => $content,
            'message_type' => $type,
            'metadata' => $metadata,
        ]);

        // Update last message timestamp
        $this->update([
            'last_message_at' => now(),
            'message_count' => $this->message_count + 1,
        ]);

        return $message;
    }

    /**
     * Mark messages as read for user
     */
    public function markAsRead(User $user, ?Message $lastMessage = null): void
    {
        $this->participants()->updateExistingPivot($user->id, [
            'last_read_at' => now(),
        ]);

        if ($lastMessage) {
            $lastMessage->markAsRead($user);
        }
    }

    /**
     * Get unread message count for user
     */
    public function getUnreadCount(User $user): int
    {
        $participant = $this->participants()->where('user_id', $user->id)->first();

        if (!$participant || !$participant->pivot->last_read_at) {
            return $this->messages()->count();
        }

        return $this->messages()
            ->where('created_at', '>', $participant->pivot->last_read_at)
            ->count();
    }

    /**
     * Check if user is participant
     */
    public function hasParticipant(User $user): bool
    {
        return $this->participants()
            ->where('user_id', $user->id)
            ->where('is_active', true)
            ->exists();
    }
}

