<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ConfigurationVersion extends Model
{
    use HasFactory;

    protected $table = 'configuration_versions';

    protected $fillable = [
        'created_by',
        'configurations',
        'version_name',
        'description',
        'is_backup',
        'is_current',
        'applied_at',
    ];

    protected $casts = [
        'configurations' => 'array',
        'is_backup' => 'boolean',
        'is_current' => 'boolean',
        'applied_at' => 'datetime',
    ];

    /**
     * User who created this version
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Mark this version as current
     */
    public function markAsCurrent(): void
    {
        // Unmark previous current version
        self::where('is_current', true)->update(['is_current' => false]);

        // Mark this as current
        $this->update([
            'is_current' => true,
            'applied_at' => now(),
        ]);
    }

    /**
     * Get differences between this and another version
     */
    public function getDiffWith(ConfigurationVersion $other): array
    {
        $diffs = [];
        $thisConfigs = $this->configurations ?? [];
        $otherConfigs = $other->configurations ?? [];

        $allKeys = array_unique(array_merge(
            array_keys($thisConfigs),
            array_keys($otherConfigs)
        ));

        foreach ($allKeys as $key) {
            $thisValue = $thisConfigs[$key] ?? null;
            $otherValue = $otherConfigs[$key] ?? null;

            if ($thisValue !== $otherValue) {
                $diffs[$key] = [
                    'from' => $otherValue,
                    'to' => $thisValue,
                ];
            }
        }

        return $diffs;
    }

    /**
     * Scope for backups
     */
    public function scopeBackups($query)
    {
        return $query->where('is_backup', true);
    }

    /**
     * Scope for snapshots
     */
    public function scopeSnapshots($query)
    {
        return $query->where('is_backup', false);
    }

    /**
     * Scope to get current version
     */
    public function scopeCurrent($query)
    {
        return $query->where('is_current', true)->first();
    }

    /**
     * Get latest version
     */
    public static function getLatest()
    {
        return self::orderBy('created_at', 'desc')->first();
    }

    /**
     * Get current active version
     */
    public static function getCurrentVersion()
    {
        return self::where('is_current', true)->first();
    }
}

