<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Faq extends Model
{
    protected $fillable = [
        'question',
        'question_ar',
        'answer',
        'answer_ar',
        'category',
        'order',
        'is_active',
        'views',
        'helpful_count',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Increment views
     */
    public function incrementViews(): void
    {
        $this->increment('views');
    }

    /**
     * Mark as helpful
     */
    public function markAsHelpful(): void
    {
        $this->increment('helpful_count');
    }

    /**
     * Get active FAQs
     */
    public static function active()
    {
        return self::where('is_active', true)->orderBy('order')->get();
    }

    /**
     * Get by category
     */
    public static function byCategory(string $category)
    {
        return self::where('category', $category)
            ->where('is_active', true)
            ->orderBy('order')
            ->get();
    }

    /**
     * Get all categories
     */
    public static function categories(): array
    {
        return [
            'general' => 'General',
            'account' => 'Account',
            'subscription' => 'Subscription',
            'offers' => 'Offers',
            'business' => 'Business',
            'payment' => 'Payment',
            'privacy' => 'Privacy & Security',
        ];
    }

    /**
     * Search FAQs
     */
    public static function search(string $query)
    {
        return self::where('is_active', true)
            ->where(function ($q) use ($query) {
                $q->where('question', 'LIKE', "%{$query}%")
                    ->orWhere('answer', 'LIKE', "%{$query}%")
                    ->orWhere('question_ar', 'LIKE', "%{$query}%")
                    ->orWhere('answer_ar', 'LIKE', "%{$query}%");
            })
            ->orderBy('order')
            ->get();
    }
}
