<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class MarketingCampaign extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'business_id',
        'subscription_id',
        'package_id',
        'name',
        'description',
        'type',
        'target_audience',
        'budget',
        'currency',
        'start_date',
        'end_date',
        'status',
        'banner_image',
        'approved_by',
        'approved_at',
        'rejection_reason',
        'targeting_options',
        'creative_assets',
        'goals',
        'metrics',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'budget' => 'decimal:2',
        'targeting_options' => 'array',
        'creative_assets' => 'array',
        'goals' => 'array',
        'metrics' => 'array',
    ];

    /**
     * Business relationship
     */
    public function business(): BelongsTo
    {
        return $this->belongsTo(Business::class);
    }

    /**
     * Subscription relationship
     */
    public function subscription(): BelongsTo
    {
        return $this->belongsTo(Subscription::class);
    }

    /**
     * Marketing package relationship
     */
    public function package(): BelongsTo
    {
        return $this->belongsTo(MarketingPackage::class, 'package_id');
    }

    /**
     * Venue relationship (if campaign is targeting specific venue)
     */
    public function venue(): BelongsTo
    {
        return $this->belongsTo(Venue::class);
    }

    /**
     * Offer relationship (if campaign is for specific offer)
     */
    public function offer(): BelongsTo
    {
        return $this->belongsTo(Offer::class);
    }

    /**
     * Analytics relationship
     */
    public function analytics(): HasMany
    {
        return $this->hasMany(MarketingCampaignAnalytic::class, 'campaign_id');
    }

    /**
     * Scope for active campaigns
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'active')
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now());
    }

    /**
     * Scope for business
     */
    public function scopeForBusiness($query, $businessId)
    {
        return $query->where('business_id', $businessId);
    }

    /**
     * Check if campaign is active
     */
    public function isActive(): bool
    {
        return $this->status === 'active'
            && $this->start_date <= now()
            && $this->end_date >= now();
    }

    /**
     * Check if campaign has ended
     */
    public function hasEnded(): bool
    {
        return $this->end_date < now();
    }

    /**
     * Get total impressions
     */
    public function getTotalImpressionsAttribute(): int
    {
        return $this->analytics()->sum('impressions');
    }

    /**
     * Get total clicks
     */
    public function getTotalClicksAttribute(): int
    {
        return $this->analytics()->sum('clicks');
    }

    /**
     * Get total conversions
     */
    public function getTotalConversionsAttribute(): int
    {
        return $this->analytics()->sum('conversions');
    }

    /**
     * Get average CTR
     */
    public function getAverageCtrAttribute(): float
    {
        $totalImpressions = $this->total_impressions;
        $totalClicks = $this->total_clicks;

        if ($totalImpressions === 0) {
            return 0;
        }

        return round(($totalClicks / $totalImpressions) * 100, 2);
    }

    /**
     * Get conversion rate
     */
    public function getConversionRateAttribute(): float
    {
        $totalClicks = $this->total_clicks;
        $totalConversions = $this->total_conversions;

        if ($totalClicks === 0) {
            return 0;
        }

        return round(($totalConversions / $totalClicks) * 100, 2);
    }

    /**
     * Get days remaining
     */
    public function getDaysRemainingAttribute(): int
    {
        if ($this->hasEnded()) {
            return 0;
        }

        return now()->diffInDays($this->end_date, false);
    }

    /**
     * Get budget spent (mock - would integrate with ad platforms)
     */
    public function getBudgetSpentAttribute(): float
    {
        // This would typically integrate with actual ad platform APIs
        // For now, return a calculated estimate based on impressions/clicks
        return round($this->total_clicks * 0.50, 2); // $0.50 per click estimate
    }

    /**
     * Get budget remaining
     */
    public function getBudgetRemainingAttribute(): float
    {
        return max(0, $this->budget - $this->budget_spent);
    }

    /**
     * Get performance percentage
     */
    public function getPerformanceAttribute(): string
    {
        // Calculate based on goals vs actuals
        $goals = $this->goals ?? [];
        $targetConversions = $goals['conversions'] ?? 0;

        if ($targetConversions === 0) {
            return 'N/A';
        }

        $percentage = ($this->total_conversions / $targetConversions) * 100;

        if ($percentage >= 100) {
            return 'excellent';
        } elseif ($percentage >= 75) {
            return 'good';
        } elseif ($percentage >= 50) {
            return 'average';
        } else {
            return 'poor';
        }
    }
}
