<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Message extends Model
{
    use HasFactory;

    protected $fillable = [
        'chat_id',
        'sender_id',
        'content',
        'message_type',
        'metadata',
        'is_edited',
        'edited_at',
        'is_deleted',
    ];

    protected $casts = [
        'metadata' => 'array',
        'is_edited' => 'boolean',
        'is_deleted' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'edited_at' => 'datetime',
    ];

    // =============== Relationships ===============

    public function chat(): BelongsTo
    {
        return $this->belongsTo(Chat::class);
    }

    public function sender(): BelongsTo
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    public function reads(): HasMany
    {
        return $this->hasMany(MessageRead::class);
    }

    // =============== Scopes ===============

    public function scopeNotDeleted($query)
    {
        return $query->where('is_deleted', false);
    }

    public function scopeText($query)
    {
        return $query->where('message_type', 'text');
    }

    // =============== Methods ===============

    /**
     * Mark message as read by user
     */
    public function markAsRead(User $user): void
    {
        MessageRead::firstOrCreate([
            'message_id' => $this->id,
            'user_id' => $user->id,
        ], [
            'read_at' => now(),
        ]);
    }

    /**
     * Get read count
     */
    public function getReadCount(): int
    {
        return $this->reads()->count();
    }

    /**
     * Check if read by user
     */
    public function isReadBy(User $user): bool
    {
        return $this->reads()
            ->where('user_id', $user->id)
            ->exists();
    }

    /**
     * Edit message
     */
    public function edit(string $newContent): void
    {
        $this->update([
            'content' => $newContent,
            'is_edited' => true,
            'edited_at' => now(),
        ]);
    }

    /**
     * Delete message (soft delete)
     */
    public function deleteMessage(): void
    {
        $this->update([
            'is_deleted' => true,
            'content' => '[Message deleted]',
        ]);
    }

    /**
     * Get preview of message (for notifications)
     */
    public function getPreview(int $length = 100): string
    {
        if ($this->is_deleted) {
            return '[Message deleted]';
        }

        return substr($this->content, 0, $length) . (strlen($this->content) > $length ? '...' : '');
    }
}

