<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Offer extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'business_id',
        'venue_id',
        'category_id',
        'title',
        'title_ar',
        'slug',
        'description',
        'description_ar',
        'offer_type',
        'discount_type',
        'discount_value',
        'original_price',
        'discounted_price',
        'terms_conditions',
        'terms_conditions_ar',
        'images',
        'featured_image',
        'start_date',
        'end_date',
        'redemption_limit',
        'max_redemptions_per_user',
        'max_redemptions_per_user_per_day',
        'daily_redemption_limit',
        'redemption_count',
        'view_count',
        'status',
        'approval_status',
        'approved_by',
        'approved_at',
        'rejection_reason',
        'is_featured',
        'is_boosted',
        'boost_end_date',
    ];

    protected $casts = [
        'images' => 'array',
        'start_date' => 'date',
        'end_date' => 'date',
        'discount_value' => 'decimal:2',
        'original_price' => 'decimal:2',
        'discounted_price' => 'decimal:2',
        'approved_at' => 'datetime',
        'is_featured' => 'boolean',
        'is_boosted' => 'boolean',
        'boost_end_date' => 'date',
    ];

    public function business()
    {
        return $this->belongsTo(Business::class);
    }

    public function venue()
    {
        return $this->belongsTo(Venue::class);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function redemptions()
    {
        return $this->hasMany(Redemption::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function favorites()
    {
        return $this->morphMany(Favorite::class, 'favorable');
    }

    public function spotlights()
    {
        return $this->hasMany(Spotlight::class);
    }

    public function isActive()
    {
        return $this->status === 'active'
            && $this->start_date <= now()
            && $this->end_date >= now();
    }

    public function isExpired()
    {
        return $this->end_date < now();
    }

    public function canBeRedeemed()
    {
        if (!$this->isActive()) {
            return false;
        }

        if ($this->redemption_limit && $this->redemption_count >= $this->redemption_limit) {
            return false;
        }

        return true;
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($offer) {
            if (empty($offer->slug)) {
                $offer->slug = \Str::slug($offer->title) . '-' . uniqid();
            }
        });
    }
}
