<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Role extends Model
{
    protected $fillable = [
        'name',
        'display_name',
        'description',
        'type',
        'is_system',
        'guard_name',
    ];

    protected $casts = [
        'is_system' => 'boolean',
    ];

    /**
     * Permissions for this role
     * Uses Spatie Permission table naming
     */
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'role_has_permissions')
            ->withTimestamps();
    }

    /**
     * Parent role for hierarchy
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(Role::class, 'parent_role_id');
    }

    /**
     * Child roles in hierarchy
     */
    public function children(): HasMany
    {
        return $this->hasMany(Role::class, 'parent_role_id');
    }

    /**
     * Users with this role
     * Uses custom user_role table
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'user_role')
            ->withTimestamps();
    }

    /**
     * Get all permissions including inherited from parent roles
     */
    public function getAllPermissions()
    {
        $permissions = $this->permissions()->get();

        // Add parent permissions recursively
        if ($this->parent) {
            $parentPermissions = $this->parent->getAllPermissions();
            $permissions = $permissions->merge($parentPermissions)->unique('id');
        }

        return $permissions;
    }

    /**
     * Check if role has permission (including inherited)
     */
    public function hasPermission(string $permissionName): bool
    {
        // Check own permissions
        if ($this->relationLoaded('permissions')) {
            if ($this->permissions->where('name', $permissionName)->isNotEmpty()) {
                return true;
            }
        } else {
            if ($this->permissions()
                ->where('name', $permissionName)
                ->exists()) {
                return true;
            }
        }

        // Check parent permissions
        if ($this->parent && $this->parent->hasPermission($permissionName)) {
            return true;
        }

        return false;
    }

    /**
     * Get all child roles recursively
     */
    public function getAllChildren()
    {
        $children = $this->children()->get();

        foreach ($children as $child) {
            $children = $children->merge($child->getAllChildren());
        }

        return $children;
    }

    /**
     * Give permission to role
     */
    public function givePermission($permission)
    {
        if (is_string($permission)) {
            $permission = Permission::where('name', $permission)->firstOrFail();
        }

        if (!$this->hasPermission($permission->name)) {
            $this->permissions()->attach($permission->id);
        }

        return $this;
    }

    /**
     * Revoke permission from role
     */
    public function revokePermission($permission)
    {
        if (is_string($permission)) {
            $permission = Permission::where('name', $permission)->firstOrFail();
        }

        $this->permissions()->detach($permission->id);

        return $this;
    }

    /**
     * Sync permissions
     */
    public function syncPermissions(array $permissionIds)
    {
        $this->permissions()->sync($permissionIds);

        return $this;
    }

    /**
     * Check if system role (cannot be deleted)
     */
    public function isSystemRole(): bool
    {
        return $this->is_system;
    }

    /**
     * Get roles by type
     */
    public static function byType(string $type)
    {
        return self::where('type', $type)->get();
    }

    /**
     * Get admin roles
     */
    public static function adminRoles()
    {
        return self::byType('admin');
    }

    /**
     * Get business roles
     */
    public static function businessRoles()
    {
        return self::byType('business');
    }

    /**
     * Scope for active roles
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for ordered by weight
     */
    public function scopeByWeight($query)
    {
        return $query->orderBy('weight', 'desc')->orderBy('created_at', 'asc');
    }
}
