<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Spotlight extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'user_id',
        'offer_id',
        'content',
        'media_type',
        'media_urls',
        'location',
        'latitude',
        'longitude',
        'status',
        'moderation_status',
        'moderated_by',
        'moderated_at',
        'moderation_notes',
        'likes_count',
        'comments_count',
        'shares_count',
        'view_count',
    ];

    protected $casts = [
        'media_urls' => 'array',
        'moderated_at' => 'datetime',
    ];

    /**
     * Get the user who created the spotlight
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the offer associated with this spotlight
     */
    public function offer(): BelongsTo
    {
        return $this->belongsTo(Offer::class);
    }

    /**
     * Get the user who moderated this spotlight
     */
    public function moderator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'moderated_by');
    }

    /**
     * Get the likes for this spotlight
     */
    public function likes(): HasMany
    {
        return $this->hasMany(SpotlightLike::class);
    }

    /**
     * Get the comments for this spotlight
     */
    public function comments(): HasMany
    {
        return $this->hasMany(SpotlightComment::class);
    }

    /**
     * Scope to get approved spotlights
     */
    public function scopeApproved($query)
    {
        return $query->where('moderation_status', 'approved');
    }

    /**
     * Scope to get pending spotlights
     */
    public function scopePending($query)
    {
        return $query->where('moderation_status', 'pending');
    }

    /**
     * Scope to get spotlights for a specific offer
     */
    public function scopeForOffer($query, $offerId)
    {
        return $query->where('offer_id', $offerId);
    }

    /**
     * Approve a spotlight
     */
    public function approve($moderatorId = null, $notes = null)
    {
        $this->update([
            'moderation_status' => 'approved',
            'moderated_by' => $moderatorId ?? auth()->id(),
            'moderated_at' => now(),
            'moderation_notes' => $notes,
        ]);
    }

    /**
     * Reject a spotlight
     */
    public function reject($moderatorId = null, $notes = null)
    {
        $this->update([
            'moderation_status' => 'rejected',
            'moderated_by' => $moderatorId ?? auth()->id(),
            'moderated_at' => now(),
            'moderation_notes' => $notes,
        ]);
    }
}

