<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SubscriptionPlan extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'name_ar',
        'description',
        'description_ar',
        'type',
        'tier',
        'user_type',
        'duration',
        'duration_days',
        'duration_type',
        'price',
        'currency',
        'features',
        'max_venues',
        'max_offers',
        'max_staff',
        'max_photos_per_venue',
        'max_favorites',
        'max_redemptions_per_month',
        'max_reviews_per_month',
        'max_banners',
        'max_impressions',
        'monthly_campaigns',
        'storage_gb',
        'analytics_access',
        'priority_support',
        'featured_listing',
        'custom_branding',
        'api_access',
        'cashback_percentage',
        'loyalty_points_multiplier',
        'ad_free',
        'early_access_days',
        'birthday_reward',
        'vip_support',
        'ab_testing_enabled',
        'retargeting_enabled',
        'audience_insights',
        'dedicated_support',
        'sort_order',
        'is_popular',
        'is_active',
        'is_private',
        'order',
    ];

    protected $casts = [
        'price' => 'decimal:2',
        'cashback_percentage' => 'decimal:2',
        'features' => 'array',
        'is_popular' => 'boolean',
        'is_active' => 'boolean',
        'is_private' => 'boolean',
        'analytics_access' => 'boolean',
        'priority_support' => 'boolean',
        'featured_listing' => 'boolean',
        'custom_branding' => 'boolean',
        'api_access' => 'boolean',
        'ad_free' => 'boolean',
        'birthday_reward' => 'boolean',
        'vip_support' => 'boolean',
        'ab_testing_enabled' => 'boolean',
        'retargeting_enabled' => 'boolean',
        'audience_insights' => 'boolean',
        'dedicated_support' => 'boolean',
    ];

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    public function activeSubscriptions()
    {
        return $this->hasMany(Subscription::class)->where('status', 'active');
    }

    public function businessSubscriptions()
    {
        return $this->hasMany(BusinessSubscription::class, 'subscription_plan_id');
    }

    public function userSubscriptions()
    {
        return $this->hasMany(UserSubscription::class, 'subscription_plan_id');
    }

    public function payments()
    {
        return $this->hasManyThrough(
            Payment::class,
            Subscription::class,
            'subscription_plan_id',
            'subscription_id',
            'id',
            'id'
        );
    }

    public function businesses()
    {
        return $this->hasMany(Business::class);
    }

    // ... existing scopes ...

    public function scopeForMarketing($query)
    {
        return $query->where('user_type', 'marketing');
    }

    public function scopeByTier($query, $tier)
    {
        return $query->where('tier', $tier);
    }

    public function scopeBasic($query)
    {
        return $query->where('tier', 'basic');
    }

    public function scopeProfessional($query)
    {
        return $query->where('tier', 'professional');
    }

    public function scopeEnterprise($query)
    {
        return $query->where('tier', 'enterprise');
    }

    public function scopePremium($query)
    {
        return $query->where('tier', 'premium');
    }

    public function scopeVIP($query)
    {
        return $query->where('tier', 'vip');
    }

    public function getDurationInDays()
    {
        return match($this->duration) {
            'monthly' => 30,
            'quarterly' => 90,
            'yearly' => 365,
            default => 30,
        };
    }
}
