<?php

namespace App\Services;

use App\Models\AdminIpRestriction;
use App\Models\AdminSession;
use App\Models\AdminActivityLog;
use Illuminate\Support\Facades\Cache;

class AdminSecurityService
{
    /**
     * Validate admin IP access
     */
    public function validateIpAccess($userId, $ipAddress): bool
    {
        // Cache check results for 5 minutes
        $cacheKey = "admin_ip_access:{$userId}:{$ipAddress}";

        return Cache::remember($cacheKey, 300, function () use ($userId, $ipAddress) {
            return AdminIpRestriction::isIpAllowed($userId, $ipAddress);
        });
    }

    /**
     * Add IP to whitelist for user
     */
    public function whitelistIp($userId, $ipAddress, $description = null, $expiresAt = null): AdminIpRestriction
    {
        $restriction = AdminIpRestriction::create([
            'user_id' => $userId,
            'ip_address' => $ipAddress,
            'type' => 'whitelist',
            'description' => $description,
            'is_active' => true,
            'expires_at' => $expiresAt,
        ]);

        $this->clearIpCache($userId);

        return $restriction;
    }

    /**
     * Add IP to blacklist for user
     */
    public function blacklistIp($userId, $ipAddress, $description = null, $expiresAt = null): AdminIpRestriction
    {
        $restriction = AdminIpRestriction::create([
            'user_id' => $userId,
            'ip_address' => $ipAddress,
            'type' => 'blacklist',
            'description' => $description,
            'is_active' => true,
            'expires_at' => $expiresAt,
        ]);

        $this->clearIpCache($userId);

        return $restriction;
    }

    /**
     * Remove IP restriction
     */
    public function removeIpRestriction($restrictionId): bool
    {
        $restriction = AdminIpRestriction::findOrFail($restrictionId);
        $userId = $restriction->user_id;

        $restriction->delete();
        $this->clearIpCache($userId);

        return true;
    }

    /**
     * Check for suspicious login activity
     */
    public function checkSuspiciousActivity($userId, $ipAddress, $userAgent): array
    {
        $warnings = [];

        // Get last 10 sessions
        $recentSessions = AdminSession::where('user_id', $userId)
            ->orderBy('login_at', 'desc')
            ->limit(10)
            ->get();

        if ($recentSessions->isEmpty()) {
            return $warnings;
        }

        // Check for multiple IPs in short time
        $lastSession = $recentSessions->first();
        $uniqueIpCount = $recentSessions->pluck('ip_address')->unique()->count();

        if ($uniqueIpCount > 5 && $recentSessions->count() >= 5) {
            $timeSpan = now()->diffInMinutes($lastSession->login_at);
            if ($timeSpan < 60) {
                $warnings[] = 'Multiple IPs detected in short time period';
            }
        }

        // Check for unusual user agent
        if ($lastSession->user_agent !== $userAgent) {
            $warnings[] = 'Different browser/device detected';
        }

        // Check for geographic impossibility (simplified - would need IP geolocation)
        // This is a placeholder for more sophisticated detection

        return $warnings;
    }

    /**
     * Record admin session
     */
    public function recordSession($userId, $ipAddress, $userAgent, $loginMethod = 'email'): AdminSession
    {
        return AdminSession::create([
            'user_id' => $userId,
            'ip_address' => $ipAddress,
            'user_agent' => $userAgent,
            'session_token' => bin2hex(random_bytes(32)),
            'login_at' => now(),
            'last_activity_at' => now(),
            'is_active' => true,
            'login_method' => $loginMethod,
            'metadata' => [
                'browser' => $this->extractBrowser($userAgent),
                'device' => $this->extractDevice($userAgent),
            ],
        ]);
    }

    /**
     * End admin session
     */
    public function endSession($sessionId): void
    {
        $session = AdminSession::findOrFail($sessionId);
        $session->logout();
    }

    /**
     * Check session is still valid
     */
    public function isSessionValid($sessionId): bool
    {
        $session = AdminSession::find($sessionId);

        if (!$session || !$session->isActive()) {
            return false;
        }

        // Check for session timeout (30 minutes of inactivity)
        if ($session->getIdleSeconds() > 1800) {
            $session->logout();
            return false;
        }

        return true;
    }

    /**
     * Update session activity
     */
    public function touchSession($sessionId): void
    {
        $session = AdminSession::find($sessionId);
        if ($session) {
            $session->touchActivity();
        }
    }

    /**
     * Get active sessions for user
     */
    public function getUserActiveSessions($userId)
    {
        return AdminSession::forUser($userId)->active()->get();
    }

    /**
     * Terminate all user sessions except current
     */
    public function terminateOtherSessions($userId, $excludeSessionId = null): int
    {
        $query = AdminSession::forUser($userId)->active();

        if ($excludeSessionId) {
            $query->where('id', '!=', $excludeSessionId);
        }

        $count = 0;
        foreach ($query->get() as $session) {
            $session->logout();
            $count++;
        }

        return $count;
    }

    /**
     * Clear IP cache
     */
    protected function clearIpCache($userId): void
    {
        // Clear all IP-related caches for this user
        Cache::tags(["admin_ip:{$userId}"])->flush();
    }

    /**
     * Extract browser from user agent
     */
    protected function extractBrowser($userAgent): string
    {
        if (strpos($userAgent, 'Chrome') !== false) {
            return 'Chrome';
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            return 'Firefox';
        } elseif (strpos($userAgent, 'Safari') !== false) {
            return 'Safari';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            return 'Edge';
        }

        return 'Unknown';
    }

    /**
     * Extract device from user agent
     */
    protected function extractDevice($userAgent): string
    {
        if (strpos($userAgent, 'Mobile') !== false) {
            return 'Mobile';
        } elseif (strpos($userAgent, 'Tablet') !== false) {
            return 'Tablet';
        }

        return 'Desktop';
    }

    /**
     * Log admin activity
     */
    public function logActivity(
        $adminUserId,
        $action,
        $entityType,
        $entityId = null,
        $entityName = null,
        $oldValues = null,
        $newValues = null,
        $description = null,
        $isSensitive = false
    ): AdminActivityLog {
        return AdminActivityLog::createFromRequest(
            $adminUserId,
            $action,
            $entityType,
            $entityId,
            $entityName,
            $oldValues,
            $newValues,
            $description,
            $isSensitive
        );
    }

    /**
     * Check for rate limiting
     */
    public function checkRateLimit($userId, $action, $limit = 10, $seconds = 60): bool
    {
        $cacheKey = "admin_rate_limit:{$userId}:{$action}";

        $count = Cache::get($cacheKey, 0);

        if ($count >= $limit) {
            return false;
        }

        Cache::increment($cacheKey);
        Cache::expire($cacheKey, $seconds);

        return true;
    }

    /**
     * Get rate limit status
     */
    public function getRateLimitStatus($userId, $action, $limit = 10): array
    {
        $cacheKey = "admin_rate_limit:{$userId}:{$action}";
        $count = Cache::get($cacheKey, 0);

        return [
            'current' => $count,
            'limit' => $limit,
            'remaining' => max(0, $limit - $count),
            'exceeded' => $count >= $limit,
        ];
    }
}

