<?php

namespace App\Services;

use App\Models\User;
use App\Models\Role;
use App\Models\BulkOperation;
use App\Models\AdminActivityLog;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class AdminUserManagementService
{
    protected $activityService;

    public function __construct()
    {
        $this->activityService = new AdminSecurityService();
    }

    /**
     * Create new user with audit trail
     */
    public function createUser(array $data, $createdById): User
    {
        $user = User::create([
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'email' => $data['email'],
            'phone' => $data['phone'] ?? null,
            'password' => Hash::make($data['password']),
            'account_type' => $data['account_type'] ?? 'user',
            'status' => $data['status'] ?? 'active',
            'email_verified_at' => $data['email_verified_at'] ?? now(),
        ]);

        // Assign roles
        if (isset($data['roles']) && is_array($data['roles'])) {
            $user->assignRole($data['roles']);
        }

        // Log activity
        AdminActivityLog::createFromRequest(
            $createdById,
            'create',
            'User',
            $user->id,
            $user->name,
            description: "Created user: {$user->email}"
        );

        return $user;
    }

    /**
     * Update user with audit trail
     */
    public function updateUser(User $user, array $data, $updatedById): User
    {
        $oldValues = $user->toArray();

        // Update basic fields
        $updateData = [];
        foreach (['first_name', 'last_name', 'email', 'phone', 'status', 'account_type'] as $field) {
            if (isset($data[$field])) {
                $updateData[$field] = $data[$field];
            }
        }

        if (!empty($updateData)) {
            $user->update($updateData);
        }

        // Update password if provided
        if (isset($data['password'])) {
            $user->update(['password' => Hash::make($data['password'])]);
        }

        // Sync roles if provided
        if (isset($data['roles']) && is_array($data['roles'])) {
            $user->syncRoles($data['roles']);
        }

        // Log activity
        AdminActivityLog::createFromRequest(
            $updatedById,
            'update',
            'User',
            $user->id,
            $user->email,
            $oldValues,
            $user->fresh()->toArray(),
            description: "Updated user: {$user->email}"
        );

        return $user->fresh();
    }

    /**
     * Delete user with audit trail
     */
    public function deleteUser(User $user, $deletedById): bool
    {
        $userData = $user->toArray();

        // Perform soft delete
        $user->delete();

        // Log activity
        AdminActivityLog::createFromRequest(
            $deletedById,
            'delete',
            'User',
            $user->id,
            $user->email,
            $userData,
            description: "Deleted user: {$user->email}",
            isSensitive: true
        );

        return true;
    }

    /**
     * Bulk update user status
     */
    public function bulkUpdateStatus(array $userIds, $newStatus, $updatedById): BulkOperation
    {
        $operation = BulkOperation::create([
            'initiated_by' => $updatedById,
            'operation_type' => 'bulk_user_status_change',
            'entity_type' => 'User',
            'total_records' => count($userIds),
            'status' => 'pending',
            'filters' => ['user_ids' => $userIds],
            'changes' => ['status' => $newStatus],
        ]);

        DB::beginTransaction();

        try {
            $operation->markAsStarted();

            $users = User::whereIn('id', $userIds)->get();
            $processedCount = 0;

            foreach ($users as $user) {
                $oldStatus = $user->status;
                $user->update(['status' => $newStatus]);

                AdminActivityLog::createFromRequest(
                    $updatedById,
                    'bulk_update',
                    'User',
                    $user->id,
                    $user->email,
                    ['status' => $oldStatus],
                    ['status' => $newStatus],
                    description: "Bulk status update: {$oldStatus} → {$newStatus}"
                );

                $processedCount++;
                $operation->updateProgress($processedCount);
            }

            $operation->markAsCompleted();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $operation->markAsFailed($e->getMessage());
        }

        return $operation;
    }

    /**
     * Bulk assign roles to users
     */
    public function bulkAssignRoles(array $userIds, array $roleIds, $assignedById): BulkOperation
    {
        $operation = BulkOperation::create([
            'initiated_by' => $assignedById,
            'operation_type' => 'bulk_role_assignment',
            'entity_type' => 'User',
            'total_records' => count($userIds),
            'status' => 'pending',
            'filters' => ['user_ids' => $userIds],
            'changes' => ['roles' => $roleIds],
        ]);

        DB::beginTransaction();

        try {
            $operation->markAsStarted();

            $users = User::whereIn('id', $userIds)->get();
            $processedCount = 0;

            foreach ($users as $user) {
                $oldRoles = $user->roles()->pluck('id')->toArray();
                $user->syncRoles($roleIds);

                AdminActivityLog::createFromRequest(
                    $assignedById,
                    'bulk_update',
                    'User',
                    $user->id,
                    $user->email,
                    ['roles' => $oldRoles],
                    ['roles' => $roleIds],
                    description: "Bulk role assignment for user: {$user->email}"
                );

                $processedCount++;
                $operation->updateProgress($processedCount);
            }

            $operation->markAsCompleted();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $operation->markAsFailed($e->getMessage());
        }

        return $operation;
    }

    /**
     * Restore deleted user
     */
    public function restoreUser(int $userId, $restoredById): User
    {
        $user = User::withTrashed()->findOrFail($userId);

        if (!$user->trashed()) {
            throw new \Exception('User is not deleted');
        }

        $user->restore();

        AdminActivityLog::createFromRequest(
            $restoredById,
            'restore',
            'User',
            $user->id,
            $user->email,
            description: "Restored deleted user: {$user->email}"
        );

        return $user;
    }

    /**
     * Get user activity summary
     */
    public function getUserActivitySummary(User $user)
    {
        return [
            'total_logins' => $user->adminSessions()->count(),
            'last_login' => $user->adminSessions()->latest('login_at')->first()?->login_at,
            'active_sessions' => $user->adminSessions()->active()->count(),
            'recent_activities' => AdminActivityLog::where('admin_user_id', $user->id)
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get(),
            'sensitive_activities' => AdminActivityLog::where('admin_user_id', $user->id)
                ->where('is_sensitive', true)
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get(),
        ];
    }

    /**
     * Check user permissions
     */
    public function getUserPermissions(User $user)
    {
        return $user->getAllPermissions()->groupBy('group');
    }

    /**
     * Export user data
     */
    public function exportUserData(User $user)
    {
        return [
            'user' => $user,
            'roles' => $user->roles()->get(),
            'permissions' => $user->getAllPermissions(),
            'sessions' => $user->adminSessions()->limit(20)->get(),
            'activity_logs' => AdminActivityLog::where('admin_user_id', $user->id)
                ->orderBy('created_at', 'desc')
                ->limit(50)
                ->get(),
        ];
    }

    /**
     * Search users
     */
    public function searchUsers($query, $limit = 20)
    {
        return User::where(function ($q) use ($query) {
            $q->where('first_name', 'like', "%{$query}%")
              ->orWhere('last_name', 'like', "%{$query}%")
              ->orWhere('email', 'like', "%{$query}%")
              ->orWhere('phone', 'like', "%{$query}%");
        })->limit($limit)->get();
    }

    /**
     * Get user statistics
     */
    public function getUserStatistics()
    {
        return [
            'total_users' => User::count(),
            'active_users' => User::where('status', 'active')->count(),
            'suspended_users' => User::where('status', 'suspended')->count(),
            'deleted_users' => User::onlyTrashed()->count(),
            'admins' => User::role('admin')->count(),
            'super_admins' => User::role('super_admin')->count(),
            'business_owners' => User::role('business_owner')->count(),
            'new_this_month' => User::whereMonth('created_at', now()->month)->count(),
            'new_this_week' => User::where('created_at', '>=', now()->subWeek())->count(),
        ];
    }
}

