<?php

namespace App\Services;

use App\Models\SystemConfiguration;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class ConfigurationService
{
    const CACHE_KEY = 'system_configuration';
    const CACHE_DURATION = 3600; // 1 hour

    /**
     * Get a configuration value
     */
    public static function get($key, $default = null)
    {
        $config = static::getAllConfigurations();
        return data_get($config, $key, $default);
    }

    /**
     * Set a configuration value
     */
    public static function set($key, $value)
    {
        try {
            $keys = explode('.', $key);
            $config = static::getAllConfigurations();

            // Navigate to the correct position and set value
            $current = &$config;
            foreach (array_slice($keys, 0, -1) as $part) {
                if (!isset($current[$part])) {
                    $current[$part] = [];
                }
                $current = &$current[$part];
            }
            $current[end($keys)] = $value;

            // Save to database
            SystemConfiguration::updateOrCreate(
                ['key' => $key],
                ['value' => is_array($value) ? json_encode($value) : $value]
            );

            // Refresh cache
            static::refreshCache();

            Log::info('Configuration updated', ['key' => $key, 'value' => $value]);
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to update configuration', ['key' => $key, 'error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Get all configurations
     */
    public static function getAllConfigurations()
    {
        return Cache::remember(static::CACHE_KEY, static::CACHE_DURATION, function () {
            $configurations = SystemConfiguration::all();
            $config = [];

            foreach ($configurations as $item) {
                $value = $item->value;

                // Try to decode JSON values
                if (is_string($value) && json_validate($value)) {
                    $value = json_decode($value, true);
                }

                data_set($config, $item->key, $value);
            }

            return $config;
        });
    }

    /**
     * Check if feature is enabled
     */
    public static function isEnabled($feature)
    {
        return (bool) static::get("features.{$feature}.enabled", false);
    }

    /**
     * Enable a feature
     */
    public static function enableFeature($feature)
    {
        return static::set("features.{$feature}.enabled", true);
    }

    /**
     * Disable a feature
     */
    public static function disableFeature($feature)
    {
        return static::set("features.{$feature}.enabled", false);
    }

    /**
     * Get email configuration
     */
    public static function getEmailConfig()
    {
        return [
            'enabled' => static::isEnabled('email'),
            'provider' => static::get('email.provider', 'smtp'),
            'from_address' => static::get('email.from_address', config('mail.from.address')),
            'from_name' => static::get('email.from_name', config('mail.from.name')),
        ];
    }

    /**
     * Get SMS configuration
     */
    public static function getSmsConfig()
    {
        return [
            'enabled' => static::isEnabled('sms'),
            'provider' => static::get('sms.provider', 'twilio'),
            'account_sid' => static::get('sms.twilio.account_sid'),
            'auth_token' => static::get('sms.twilio.auth_token'),
            'phone_number' => static::get('sms.twilio.phone_number'),
        ];
    }

    /**
     * Get payment configuration
     */
    public static function getPaymentConfig()
    {
        return [
            'enabled' => static::isEnabled('payment'),
            'provider' => static::get('payment.provider', 'stripe'),
            'currency' => static::get('payment.currency', 'USD'),
            'stripe_key' => static::get('payment.stripe.public_key'),
            'stripe_secret' => static::get('payment.stripe.secret_key'),
        ];
    }

    /**
     * Get security configuration
     */
    public static function getSecurityConfig()
    {
        return [
            'enable_2fa' => static::isEnabled('security.two_factor_auth'),
            'session_timeout' => static::get('security.session_timeout', 3600),
            'password_expiry_days' => static::get('security.password_expiry_days', 90),
            'require_password_change' => static::isEnabled('security.require_password_change'),
            'allow_api_tokens' => static::isEnabled('security.allow_api_tokens'),
        ];
    }

    /**
     * Refresh cache
     */
    public static function refreshCache()
    {
        Cache::forget(static::CACHE_KEY);
        return static::getAllConfigurations();
    }

    /**
     * Bulk update configurations
     */
    public static function bulkUpdate($configurations)
    {
        try {
            foreach ($configurations as $key => $value) {
                static::set($key, $value);
            }
            Log::info('Bulk configuration update completed', ['count' => count($configurations)]);
            return true;
        } catch (\Exception $e) {
            Log::error('Bulk configuration update failed', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Reset to defaults
     */
    public static function resetToDefaults()
    {
        try {
            SystemConfiguration::truncate();
            static::refreshCache();
            Log::info('Configuration reset to defaults');
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to reset configuration', ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Export configuration
     */
    public static function export()
    {
        return static::getAllConfigurations();
    }

    /**
     * Import configuration
     */
    public static function import($data)
    {
        try {
            foreach (array_dot($data) as $key => $value) {
                static::set($key, $value);
            }
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to import configuration', ['error' => $e->getMessage()]);
            return false;
        }
    }
}

