<?php

namespace App\Services\Notification;

use App\Models\User;
use App\Models\EmailTemplate;
use App\Models\SmsTemplate;
use App\Models\NotificationQueue;
use App\Models\SentNotification;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send notification via multiple channels
     */
    public function send(
        User $user,
        string $templateSlug,
        array $data = [],
        array $channels = ['email']
    ): bool {
        try {
            foreach ($channels as $channel) {
                match($channel) {
                    'email' => $this->sendEmail($user, $templateSlug, $data),
                    'sms' => $this->sendSms($user, $templateSlug, $data),
                    'push' => $this->sendPush($user, $templateSlug, $data),
                    default => null,
                };
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Notification send failed', [
                'user_id' => $user->id,
                'template' => $templateSlug,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Queue notification for later sending
     */
    public function queue(
        User $user,
        string $channel,
        string $templateSlug,
        array $data = [],
        ?\DateTime $sendAt = null
    ): NotificationQueue {
        return NotificationQueue::create([
            'user_id' => $user->id,
            'channel' => $channel,
            'template_slug' => $templateSlug,
            'recipient' => $this->getRecipientForChannel($user, $channel),
            'data' => $data,
            'status' => 'pending',
        ]);
    }

    /**
     * Send email notification
     */
    public function sendEmail(User $user, string $templateSlug, array $data = []): void
    {
        // Get template
        $template = EmailTemplate::where('slug', $templateSlug)
            ->where('is_active', true)
            ->firstOrFail();

        // Check user preferences
        if (!$this->shouldSendEmail($user)) {
            return;
        }

        // Render template
        $rendered = $template->render(array_merge([
            'user_name' => $user->first_name . ' ' . $user->last_name,
            'email' => $user->email,
        ], $data));

        // Send email
        try {
            Mail::send('emails.generic', [
                'subject' => $rendered['subject'],
                'body' => $rendered['body'],
            ], function ($message) use ($user, $rendered) {
                $message->to($user->email)
                    ->subject($rendered['subject']);
            });

            // Log sent notification
            SentNotification::create([
                'user_id' => $user->id,
                'channel' => 'email',
                'category' => $template->category,
                'subject' => $rendered['subject'],
                'content' => $rendered['body'],
                'recipient' => $user->email,
                'status' => 'sent',
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            Log::error('Email send failed', [
                'user_id' => $user->id,
                'template' => $templateSlug,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Send SMS notification
     */
    public function sendSms(User $user, string $templateSlug, array $data = []): void
    {
        if (!$user->phone) {
            return;
        }

        // Get template
        $template = SmsTemplate::where('slug', $templateSlug)
            ->where('is_active', true)
            ->firstOrFail();

        // Check user preferences
        if (!$this->shouldSendSms($user)) {
            return;
        }

        // Render template
        $content = $template->render(array_merge([
            'user_name' => $user->first_name,
        ], $data));

        // Send via SMS provider (Twilio, etc.)
        try {
            $this->sendViaSmsProvider($user->phone, $content);

            // Log sent notification
            SentNotification::create([
                'user_id' => $user->id,
                'channel' => 'sms',
                'category' => $template->category,
                'content' => $content,
                'recipient' => $user->phone,
                'status' => 'sent',
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            Log::error('SMS send failed', [
                'user_id' => $user->id,
                'template' => $templateSlug,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Send push notification
     */
    public function sendPush(User $user, string $templateSlug, array $data = []): void
    {
        if (!$user->fcm_token) {
            return;
        }

        // Check user preferences
        if (!$this->shouldSendPush($user)) {
            return;
        }

        try {
            // Send via Firebase Cloud Messaging
            $this->sendViaFirebase($user->fcm_token, $data);

            // Log sent notification
            SentNotification::create([
                'user_id' => $user->id,
                'channel' => 'push',
                'category' => $data['category'] ?? 'system',
                'subject' => $data['title'] ?? 'Notification',
                'content' => $data['body'] ?? '',
                'recipient' => $user->fcm_token,
                'status' => 'sent',
                'sent_at' => now(),
            ]);
        } catch (\Exception $e) {
            Log::error('Push send failed', [
                'user_id' => $user->id,
                'template' => $templateSlug,
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Check if email should be sent to user
     */
    public function shouldSendEmail(User $user): bool
    {
        $preferences = $user->notificationPreferences;

        return $preferences?->email_enabled ?? true;
    }

    /**
     * Check if SMS should be sent to user
     */
    public function shouldSendSms(User $user): bool
    {
        $preferences = $user->notificationPreferences;

        return $preferences?->sms_enabled ?? false;
    }

    /**
     * Check if push should be sent to user
     */
    public function shouldSendPush(User $user): bool
    {
        $preferences = $user->notificationPreferences;

        return $preferences?->push_enabled ?? true;
    }

    /**
     * Get recipient address for channel
     */
    private function getRecipientForChannel(User $user, string $channel): string
    {
        return match($channel) {
            'email' => $user->email,
            'sms' => $user->phone ?? '',
            'push' => $user->fcm_token ?? '',
            default => '',
        };
    }

    /**
     * Send via SMS provider (placeholder)
     */
    private function sendViaSmsProvider(string $phone, string $message): void
    {
        // TODO: Implement with Twilio/Nexmo/etc.
        // Example:
        // $twilio = new Client(config('services.twilio.account_sid'), config('services.twilio.auth_token'));
        // $twilio->messages->create($phone, ['from' => config('services.twilio.from_number'), 'body' => $message]);

        Log::info('SMS would be sent', ['phone' => $phone, 'message' => $message]);
    }

    /**
     * Send via Firebase Cloud Messaging (placeholder)
     */
    private function sendViaFirebase(string $token, array $data): void
    {
        // TODO: Implement with Firebase
        // Example using kreait/firebase-php library
        // $messaging = app('firebase.messaging');
        // $message = CloudMessage::withData($data)->withWebsafeBoundary();
        // $messaging->send($message, $token);

        Log::info('Push would be sent', ['token' => $token, 'data' => $data]);
    }

    /**
     * Process queued notifications
     */
    public function processQueue(): void
    {
        $queued = NotificationQueue::where('status', 'pending')
            ->orWhere('status', 'retry')
            ->get();

        foreach ($queued as $item) {
            try {
                $this->send(
                    $item->user,
                    $item->template_slug,
                    $item->data,
                    [$item->channel]
                );

                $item->markAsSent();
            } catch (\Exception $e) {
                if ($item->retry_count < 3) {
                    $item->markForRetry($e->getMessage());
                } else {
                    $item->markAsFailed($e->getMessage());
                }
            }
        }
    }
}

