<?php

namespace App\Services\Payment;

use App\Models\Payment;
use Srmklive\PayPal\Services\PayPal as PayPalClient;

class PayPalService
{
    protected $provider;

    public function __construct()
    {
        $this->provider = new PayPalClient;
        $this->provider->setApiCredentials(config('paypal'));
        $this->provider->setAccessToken($this->provider->getAccessToken());
    }

    public function createOrder($amount, $currency = 'USD', $description = 'Subscription Payment')
    {
        try {
            $order = $this->provider->createOrder([
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'amount' => [
                        'currency_code' => $currency,
                        'value' => number_format($amount, 2, '.', ''),
                    ],
                    'description' => $description,
                ]],
                'application_context' => [
                    'return_url' => route('paypal.success'),
                    'cancel_url' => route('paypal.cancel'),
                ],
            ]);

            if (isset($order['id'])) {
                return [
                    'success' => true,
                    'order_id' => $order['id'],
                    'approval_url' => $this->getApprovalUrl($order),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to create PayPal order',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function captureOrder($orderId)
    {
        try {
            $result = $this->provider->capturePaymentOrder($orderId);

            if (isset($result['status']) && $result['status'] === 'COMPLETED') {
                return [
                    'success' => true,
                    'transaction_id' => $result['id'],
                    'data' => $result,
                ];
            }

            return [
                'success' => false,
                'error' => 'Payment not completed',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function createSubscription($planId)
    {
        try {
            $subscription = $this->provider->createSubscription([
                'plan_id' => $planId,
                'application_context' => [
                    'return_url' => route('paypal.subscription.success'),
                    'cancel_url' => route('paypal.subscription.cancel'),
                ],
            ]);

            if (isset($subscription['id'])) {
                return [
                    'success' => true,
                    'subscription_id' => $subscription['id'],
                    'approval_url' => $this->getApprovalUrl($subscription),
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to create subscription',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function cancelSubscription($subscriptionId, $reason = '')
    {
        try {
            $result = $this->provider->cancelSubscription($subscriptionId, $reason);

            return ['success' => true];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function handleWebhook($payload)
    {
        try {
            $eventType = $payload['event_type'] ?? null;

            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                    $this->handlePaymentCaptured($payload);
                    break;

                case 'BILLING.SUBSCRIPTION.ACTIVATED':
                    $this->handleSubscriptionActivated($payload);
                    break;

                case 'BILLING.SUBSCRIPTION.CANCELLED':
                    $this->handleSubscriptionCancelled($payload);
                    break;

                case 'BILLING.SUBSCRIPTION.EXPIRED':
                    $this->handleSubscriptionExpired($payload);
                    break;
            }

            return ['success' => true];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    protected function getApprovalUrl($order)
    {
        foreach ($order['links'] as $link) {
            if ($link['rel'] === 'approve') {
                return $link['href'];
            }
        }
        return null;
    }

    protected function handlePaymentCaptured($payload)
    {
        // Create payment record
        // TODO: Implement payment recording logic
    }

    protected function handleSubscriptionActivated($payload)
    {
        // Update subscription status
        // TODO: Implement subscription activation logic
    }

    protected function handleSubscriptionCancelled($payload)
    {
        // Update subscription status
        // TODO: Implement cancellation logic
    }

    protected function handleSubscriptionExpired($payload)
    {
        // Update subscription status
        // TODO: Implement expiration logic
    }

    public function refund($captureId, $amount = null)
    {
        try {
            $refundData = [];

            if ($amount) {
                $refundData['amount'] = [
                    'value' => number_format($amount, 2, '.', ''),
                    'currency_code' => 'USD',
                ];
            }

            $result = $this->provider->refundCapturedPayment($captureId, $refundData);

            if (isset($result['status']) && $result['status'] === 'COMPLETED') {
                return [
                    'success' => true,
                    'refund_id' => $result['id'],
                ];
            }

            return [
                'success' => false,
                'error' => 'Refund failed',
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}
