<?php

namespace App\Services\SMS;

use App\Models\SystemConfiguration;

class SMSService
{
    protected $provider;

    public function __construct()
    {
        $this->provider = $this->getProvider();
    }

    /**
     * Get SMS provider based on configuration
     */
    protected function getProvider(): SMSProviderInterface
    {
        $providerName = SystemConfiguration::getSmsProvider();

        switch ($providerName) {
            case 'twilio':
                return new TwilioService();
            case 'sms_global':
                return new SMSGlobalService();
            case 'etisalat':
                return new EtisalatService();
            default:
                return new TwilioService();
        }
    }

    /**
     * Send SMS
     */
    public function send(string $to, string $message): array
    {
        // Check if SMS is enabled
        if (!SystemConfiguration::isEnabled('notification.sms_enabled')) {
            return [
                'success' => false,
                'error' => 'SMS notifications are disabled',
            ];
        }

        return $this->provider->send($to, $message);
    }

    /**
     * Send OTP
     */
    public function sendOTP(string $to, string $otp): array
    {
        return $this->provider->sendOTP($to, $otp);
    }

    /**
     * Get message status
     */
    public function getStatus(string $messageId): array
    {
        return $this->provider->getStatus($messageId);
    }
}
