<?php

namespace App\Services\SMS;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class TwilioService implements SMSProviderInterface
{
    protected $accountSid;
    protected $authToken;
    protected $fromNumber;

    public function __construct()
    {
        $this->accountSid = config('services.twilio.account_sid');
        $this->authToken = config('services.twilio.auth_token');
        $this->fromNumber = config('services.twilio.from');
    }

    /**
     * Send SMS
     */
    public function send(string $to, string $message): array
    {
        try {
            $response = Http::asForm()
                ->withBasicAuth($this->accountSid, $this->authToken)
                ->post("https://api.twilio.com/2010-04-01/Accounts/{$this->accountSid}/Messages.json", [
                    'From' => $this->fromNumber,
                    'To' => $to,
                    'Body' => $message,
                ]);

            if ($response->successful()) {
                $data = $response->json();

                return [
                    'success' => true,
                    'message_id' => $data['sid'] ?? null,
                    'status' => $data['status'] ?? 'sent',
                    'data' => $data,
                ];
            }

            return [
                'success' => false,
                'error' => $response->json()['message'] ?? 'Failed to send SMS',
                'status_code' => $response->status(),
            ];
        } catch (Exception $e) {
            Log::error('Twilio SMS Error: ' . $e->getMessage());

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    /**
     * Send OTP
     */
    public function sendOTP(string $to, string $otp): array
    {
        $message = "Your Privasee verification code is: {$otp}. Valid for 10 minutes.";

        return $this->send($to, $message);
    }

    /**
     * Get delivery status
     */
    public function getStatus(string $messageId): array
    {
        try {
            $response = Http::withBasicAuth($this->accountSid, $this->authToken)
                ->get("https://api.twilio.com/2010-04-01/Accounts/{$this->accountSid}/Messages/{$messageId}.json");

            if ($response->successful()) {
                $data = $response->json();

                return [
                    'success' => true,
                    'status' => $data['status'],
                    'data' => $data,
                ];
            }

            return [
                'success' => false,
                'error' => 'Failed to get message status',
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }
}
