<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('system_configurations', function (Blueprint $table) {
            $table->id();
            $table->string('key')->unique();
            $table->string('group')->default('general');
            $table->text('value')->nullable();
            $table->string('type')->default('string');
            $table->text('description')->nullable();
            $table->boolean('is_public')->default(false);
            $table->json('options')->nullable();
            $table->integer('sort_order')->default(0);
            $table->timestamps();

            $table->index(['group', 'sort_order']);
            $table->index('key');
        });

        // Insert default configurations
        $records = [
            // ---------------- Authentication ----------------
            [
                'key' => 'auth.allow_social_login',
                'group' => 'authentication',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Enable social login (Facebook, Google)',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.require_email_verification',
                'group' => 'authentication',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Require email verification for new users',
                'is_public' => false,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.require_phone_verification',
                'group' => 'authentication',
                'value' => 'false',
                'type' => 'boolean',
                'description' => 'Require phone verification for new users',
                'is_public' => false,
                'sort_order' => 3,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.allow_guest_browsing',
                'group' => 'authentication',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Allow unauthenticated users to browse venues',
                'is_public' => false,
                'sort_order' => 4,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.guest_can_view_offers',
                'group' => 'authentication',
                'value' => 'false',
                'type' => 'boolean',
                'description' => 'Allow guests to view offers',
                'is_public' => false,
                'sort_order' => 5,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.default_subscription_plan_id',
                'group' => 'authentication',
                'value' => '1',
                'type' => 'integer',
                'description' => 'Default subscription plan ID for new users',
                'is_public' => false,
                'sort_order' => 6,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'auth.default_subscription_duration_days',
                'group' => 'authentication',
                'value' => '30',
                'type' => 'integer',
                'description' => 'Default subscription duration in days',
                'is_public' => false,
                'sort_order' => 7,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Referral ----------------
            [
                'key' => 'referral.enabled',
                'group' => 'referral',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Enable referral system',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Spotlight ----------------
            [
                'key' => 'spotlight.require_moderation',
                'group' => 'spotlight',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Require admin moderation for spotlights',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'spotlight.allow_comments',
                'group' => 'spotlight',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Allow comments on spotlights',
                'is_public' => false,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'spotlight.max_per_redemption',
                'group' => 'spotlight',
                'value' => '3',
                'type' => 'integer',
                'description' => 'Maximum spotlights per offer redemption',
                'is_public' => false,
                'sort_order' => 3,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Venue ----------------
            [
                'key' => 'venue.require_approval',
                'group' => 'venue',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Require admin approval for new venues',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Offer ----------------
            [
                'key' => 'offer.require_approval',
                'group' => 'offer',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Require admin approval for new offers',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'offer.types',
                'group' => 'offer',
                'value' => json_encode(['regular', 'special', 'exclusive']),
                'type' => 'json',
                'description' => 'Available offer types',
                'is_public' => false,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Notifications ----------------
            [
                'key' => 'notification.email_enabled',
                'group' => 'notification',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Enable email notifications',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'notification.sms_enabled',
                'group' => 'notification',
                'value' => 'false',
                'type' => 'boolean',
                'description' => 'Enable SMS notifications',
                'is_public' => false,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'notification.push_enabled',
                'group' => 'notification',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Enable push notifications',
                'is_public' => false,
                'sort_order' => 3,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- SMS Provider ----------------
            [
                'key' => 'sms.provider',
                'group' => 'sms',
                'value' => 'twilio',
                'type' => 'select',
                'description' => 'SMS provider',
                'options' => json_encode(['twilio', 'sms_global', 'etisalat']),
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Languages ----------------
            [
                'key' => 'app.languages',
                'group' => 'general',
                'value' => json_encode(['en', 'ar']),
                'type' => 'json',
                'description' => 'Enabled languages',
                'is_public' => true,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'app.default_language',
                'group' => 'general',
                'value' => 'en',
                'type' => 'string',
                'description' => 'Default language',
                'is_public' => true,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- Chat ----------------
            [
                'key' => 'chat.enabled',
                'group' => 'chat',
                'value' => 'false',
                'type' => 'boolean',
                'description' => 'Enable chat feature',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],

            // ---------------- GDPR ----------------
            [
                'key' => 'gdpr.enabled',
                'group' => 'gdpr',
                'value' => 'true',
                'type' => 'boolean',
                'description' => 'Enable GDPR compliance features',
                'is_public' => false,
                'sort_order' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'key' => 'gdpr.data_retention_days',
                'group' => 'gdpr',
                'value' => '365',
                'type' => 'integer',
                'description' => 'Data retention period in days',
                'is_public' => false,
                'sort_order' => 2,
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ];

        // Ensure consistent keys (add missing 'options' => null)
        $records = array_map(function ($item) {
            if (!array_key_exists('options', $item)) {
                $item['options'] = null;
            }
            return $item;
        }, $records);

        DB::table('system_configurations')->insert($records);
    }

    public function down(): void
    {
        Schema::dropIfExists('system_configurations');
    }
};
