<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Enhance roles table with hierarchy support
        if (Schema::hasTable('roles')) {
            Schema::table('roles', function (Blueprint $table) {
                if (!Schema::hasColumn('roles', 'parent_role_id')) {
                    $table->foreignId('parent_role_id')->nullable()->after('is_system')->constrained('roles')->nullOnDelete();
                }
                if (!Schema::hasColumn('roles', 'weight')) {
                    $table->integer('weight')->default(0)->after('parent_role_id');
                }
                if (!Schema::hasColumn('roles', 'is_active')) {
                    $table->boolean('is_active')->default(true)->after('weight');
                }
            });
        }

        // Admin sessions table for tracking active admin users
        if (!Schema::hasTable('admin_sessions')) {
            Schema::create('admin_sessions', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained()->onDelete('cascade');
                $table->string('ip_address')->nullable();
                $table->string('user_agent')->nullable();
                $table->string('session_token')->unique();
                $table->timestamp('login_at');
                $table->timestamp('last_activity_at');
                $table->timestamp('logout_at')->nullable();
                $table->boolean('is_active')->default(true);
                $table->string('login_method')->default('email'); // email, mfa, token
                $table->json('metadata')->nullable(); // browser, device info
                $table->timestamps();

                $table->index('user_id');
                $table->index('is_active');
                $table->index('login_at');
            });
        }

        // Admin activity log for comprehensive auditing
        if (!Schema::hasTable('admin_activity_logs')) {
            Schema::create('admin_activity_logs', function (Blueprint $table) {
                $table->id();
                $table->foreignId('admin_user_id')->constrained('users')->onDelete('cascade');
                $table->string('action')->index(); // create, update, delete, view, export, etc.
                $table->string('entity_type')->index(); // User, Role, Permission, Configuration, etc.
                $table->unsignedBigInteger('entity_id')->nullable();
                $table->string('entity_name')->nullable();
                $table->json('old_values')->nullable();
                $table->json('new_values')->nullable();
                $table->json('changes')->nullable(); // Summary of what changed
                $table->string('ip_address')->nullable();
                $table->string('user_agent')->nullable();
                $table->text('description')->nullable();
                $table->string('status')->default('success'); // success, failed, warning
                $table->boolean('requires_approval')->default(false);
                $table->boolean('is_sensitive')->default(false);
                $table->timestamps();

                $table->index('admin_user_id');
                $table->index('is_sensitive');
                $table->index('created_at');
            });
        }

        // Two-factor authentication settings
        if (!Schema::hasTable('admin_mfa_settings')) {
            Schema::create('admin_mfa_settings', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->unique()->constrained()->onDelete('cascade');
                $table->enum('mfa_method', ['email', 'authenticator', 'sms'])->default('email');
                $table->string('secret_key')->nullable(); // For authenticator app
                $table->boolean('is_enabled')->default(false);
                $table->timestamp('enabled_at')->nullable();
                $table->json('backup_codes')->nullable();
                $table->timestamp('verified_at')->nullable();
                $table->integer('failed_attempts')->default(0);
                $table->timestamp('locked_until')->nullable();
                $table->timestamps();

                $table->index('user_id');
            });
        }

        // IP whitelist/blacklist for admin access
        if (!Schema::hasTable('admin_ip_restrictions')) {
            Schema::create('admin_ip_restrictions', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->nullable()->constrained()->onDelete('cascade');
                $table->string('ip_address');
                $table->enum('type', ['whitelist', 'blacklist']);
                $table->text('description')->nullable();
                $table->boolean('is_active')->default(true);
                $table->timestamp('expires_at')->nullable();
                $table->timestamps();

                $table->index('user_id');
                $table->index('ip_address');
                $table->index('type');
                $table->index('is_active');
            });
        }

        // Configuration versioning and backup
        if (!Schema::hasTable('configuration_versions')) {
            Schema::create('configuration_versions', function (Blueprint $table) {
                $table->id();
                $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
                $table->json('configurations');
                $table->string('version_name')->nullable();
                $table->text('description')->nullable();
                $table->boolean('is_backup')->default(false);
                $table->boolean('is_current')->default(false);
                $table->timestamp('applied_at')->nullable();
                $table->timestamps();

                $table->index('created_by');
                $table->index('created_at');
                $table->index('is_current');
            });
        }

        // Bulk operation tracking
        if (!Schema::hasTable('bulk_operations')) {
            Schema::create('bulk_operations', function (Blueprint $table) {
                $table->id();
                $table->foreignId('initiated_by')->constrained('users')->onDelete('cascade');
                $table->string('operation_type'); // bulk_user_status_change, bulk_role_assignment, etc.
                $table->string('entity_type'); // User, Role, Permission, etc.
                $table->integer('total_records');
                $table->integer('processed_records')->default(0);
                $table->string('status')->default('pending'); // pending, in_progress, completed, failed
                $table->json('filters')->nullable();
                $table->json('changes')->nullable();
                $table->text('error_message')->nullable();
                $table->timestamp('started_at')->nullable();
                $table->timestamp('completed_at')->nullable();
                $table->timestamps();

                $table->index('initiated_by');
                $table->index('status');
                $table->index('created_at');
            });
        }

        // Enhance system_configurations table with versioning
        Schema::table('system_configurations', function (Blueprint $table) {
            if (!Schema::hasColumn('system_configurations', 'version')) {
                $table->integer('version')->default(1)->after('sort_order');
            }
            if (!Schema::hasColumn('system_configurations', 'requires_approval')) {
                $table->boolean('requires_approval')->default(false)->after('version');
            }
            if (!Schema::hasColumn('system_configurations', 'changed_by')) {
                $table->foreignId('changed_by')->nullable()->after('requires_approval')->constrained('users')->nullOnDelete();
            }
            if (!Schema::hasColumn('system_configurations', 'approved_by')) {
                $table->foreignId('approved_by')->nullable()->after('changed_by')->constrained('users')->nullOnDelete();
            }
            if (!Schema::hasColumn('system_configurations', 'approved_at')) {
                $table->timestamp('approved_at')->nullable()->after('approved_by');
            }
        });
    }

    public function down(): void
    {
        // Drop new tables
        Schema::dropIfExists('bulk_operations');
        Schema::dropIfExists('configuration_versions');
        Schema::dropIfExists('admin_ip_restrictions');
        Schema::dropIfExists('admin_mfa_settings');
        Schema::dropIfExists('admin_activity_logs');
        Schema::dropIfExists('admin_sessions');

        // Remove added columns from roles
        Schema::table('roles', function (Blueprint $table) {
            if (Schema::hasColumn('roles', 'is_active')) {
                $table->dropColumn('is_active');
            }
            if (Schema::hasColumn('roles', 'weight')) {
                $table->dropColumn('weight');
            }
            if (Schema::hasColumn('roles', 'parent_role_id')) {
                $table->dropForeignKeyIfExists(['parent_role_id']);
                $table->dropColumn('parent_role_id');
            }
        });

        // Remove added columns from system_configurations
        Schema::table('system_configurations', function (Blueprint $table) {
            if (Schema::hasColumn('system_configurations', 'approved_at')) {
                $table->dropColumn('approved_at');
            }
            if (Schema::hasColumn('system_configurations', 'approved_by')) {
                $table->dropForeignKeyIfExists(['approved_by']);
                $table->dropColumn('approved_by');
            }
            if (Schema::hasColumn('system_configurations', 'changed_by')) {
                $table->dropForeignKeyIfExists(['changed_by']);
                $table->dropColumn('changed_by');
            }
            if (Schema::hasColumn('system_configurations', 'requires_approval')) {
                $table->dropColumn('requires_approval');
            }
            if (Schema::hasColumn('system_configurations', 'version')) {
                $table->dropColumn('version');
            }
        });
    }
};

