<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Email templates for admin management
        Schema::create('email_templates', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique();
            $table->string('slug')->unique();
            $table->string('category'); // account, business, team, offer, etc.
            $table->string('subject');
            $table->longText('html_body');
            $table->longText('text_body')->nullable();
            $table->json('available_variables')->nullable();
            $table->json('metadata')->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
            $table->softDeletes();

            $table->index('category');
            $table->index('slug');
            $table->index('is_active');
        });

        // SMS templates for admin management
        Schema::create('sms_templates', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique();
            $table->string('slug')->unique();
            $table->string('category');
            $table->longText('content');
            $table->json('available_variables')->nullable();
            $table->integer('character_count')->default(0);
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->nullable()->constrained('users')->onDelete('set null');
            $table->foreignId('updated_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();
            $table->softDeletes();

            $table->index('category');
            $table->index('slug');
            $table->index('is_active');
        });

        // Notification triggers/rules
        Schema::create('notification_triggers', function (Blueprint $table) {
            $table->id();
            $table->string('event_name'); // user.created, team.member_added, etc.
            $table->string('event_class'); // Full class path for event listener
            $table->string('category');
            $table->json('channels'); // ['email', 'sms', 'push']
            $table->json('conditions')->nullable(); // Rules for when to send
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->unique('event_name');
            $table->index('category');
        });

        // Notification queue (for tracking sent notifications)
        if (!Schema::hasTable('notification_queue')) {
            Schema::create('notification_queue', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained()->onDelete('cascade');
                $table->string('channel'); // email, sms, push
                $table->string('template_slug')->nullable();
                $table->string('recipient'); // email, phone, or token
                $table->json('data')->nullable();
                $table->enum('status', ['pending', 'sent', 'failed', 'retry'])->default('pending');
                $table->integer('retry_count')->default(0);
                $table->text('error_message')->nullable();
                $table->timestamp('sent_at')->nullable();
                $table->timestamps();

                $table->index(['user_id', 'status']);
                $table->index(['channel', 'status']);
                $table->index('created_at');
            });
        }

        // Sent notification history
        if (!Schema::hasTable('sent_notifications')) {
            Schema::create('sent_notifications', function (Blueprint $table) {
                $table->id();
                $table->foreignId('user_id')->constrained()->onDelete('cascade');
                $table->string('channel'); // email, sms, push
                $table->string('category');
                $table->string('subject')->nullable();
                $table->longText('content');
                $table->string('recipient');
                $table->enum('status', ['sent', 'failed', 'bounced', 'complained'])->default('sent');
                $table->json('metadata')->nullable();
                $table->timestamp('sent_at');
                $table->timestamps();

                $table->index(['user_id', 'channel']);
                $table->index(['category', 'created_at']);
                $table->index('sent_at');
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('sent_notifications');
        Schema::dropIfExists('notification_queue');
        Schema::dropIfExists('notification_triggers');
        Schema::dropIfExists('sms_templates');
        Schema::dropIfExists('email_templates');
    }
};

