<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations - Fix Spatie Permission table naming
     */
    public function up(): void
    {
        // Check if custom tables exist
        if (Schema::hasTable('role_permission')) {
            // Rename custom table to Spatie naming convention
            Schema::rename('role_permission', 'role_has_permissions');
        }

        // Ensure role_has_permissions table has correct structure for Spatie
        if (!Schema::hasTable('role_has_permissions')) {
            Schema::create('role_has_permissions', function (Blueprint $table) {
                $table->id();
                $table->foreignId('permission_id')
                    ->constrained('permissions')
                    ->onDelete('cascade');
                $table->foreignId('role_id')
                    ->constrained('roles')
                    ->onDelete('cascade');
                $table->timestamps();

                $table->unique(['permission_id', 'role_id']);
            });
        }

        // Ensure model_has_permissions table exists for Spatie (User-Permission direct relationship)
        if (!Schema::hasTable('model_has_permissions')) {
            Schema::create('model_has_permissions', function (Blueprint $table) {
                $table->id();
                $table->foreignId('permission_id')
                    ->constrained('permissions')
                    ->onDelete('cascade');
                $table->morphs('model');
                $table->timestamps();

                $table->unique(['permission_id', 'model_id', 'model_type']);
            });
        }

        // Ensure model_has_roles table exists for Spatie
        if (!Schema::hasTable('model_has_roles')) {
            Schema::create('model_has_roles', function (Blueprint $table) {
                $table->id();
                $table->foreignId('role_id')
                    ->constrained('roles')
                    ->onDelete('cascade');
                $table->morphs('model');
                $table->timestamps();

                $table->unique(['role_id', 'model_id', 'model_type']);
            });
        }

        // If user_role exists, migrate data to model_has_roles
        if (Schema::hasTable('user_role')) {
            // Copy data from user_role to model_has_roles
            $userRoles = DB::table('user_role')->get();

            foreach ($userRoles as $userRole) {
                DB::table('model_has_roles')->insertOrIgnore([
                    'role_id' => $userRole->role_id,
                    'model_id' => $userRole->user_id,
                    'model_type' => 'App\\Models\\User',
                    'created_at' => $userRole->created_at ?? now(),
                    'updated_at' => $userRole->updated_at ?? now(),
                ]);
            }
        }
    }

    /**
     * Reverse the migrations
     */
    public function down(): void
    {
        // Drop Spatie tables
        Schema::dropIfExists('model_has_roles');
        Schema::dropIfExists('model_has_permissions');

        // Rename back to custom naming
        if (Schema::hasTable('role_has_permissions')) {
            Schema::rename('role_has_permissions', 'role_permission');
        }
    }
};

