<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Role;
use App\Models\Permission;

class AdminConfigurationSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create admin role hierarchy
        $superAdmin = Role::firstOrCreate(
            ['name' => 'super_admin'],
            [
                'display_name' => 'Super Administrator',
                'description' => 'Full system access with configuration control',
                'type' => 'admin',
                'is_system' => true,
                'weight' => 100,
                'is_active' => true,
            ]
        );

        $admin = Role::firstOrCreate(
            ['name' => 'admin'],
            [
                'display_name' => 'Administrator',
                'description' => 'Platform management and moderation',
                'type' => 'admin',
                'is_system' => true,
                'weight' => 50,
                'parent_role_id' => $superAdmin->id,
                'is_active' => true,
            ]
        );

        $manager = Role::firstOrCreate(
            ['name' => 'manager'],
            [
                'display_name' => 'Manager',
                'description' => 'Team and content management',
                'type' => 'admin',
                'is_system' => true,
                'weight' => 25,
                'parent_role_id' => $admin->id,
                'is_active' => true,
            ]
        );

        // Create admin permissions
        $adminPermissions = [
            // User Management
            ['name' => 'users.view', 'display_name' => 'View Users', 'group' => 'User Management'],
            ['name' => 'users.create', 'display_name' => 'Create Users', 'group' => 'User Management'],
            ['name' => 'users.update', 'display_name' => 'Update Users', 'group' => 'User Management'],
            ['name' => 'users.delete', 'display_name' => 'Delete Users', 'group' => 'User Management'],
            ['name' => 'users.bulk_update', 'display_name' => 'Bulk Update Users', 'group' => 'User Management'],
            ['name' => 'users.export', 'display_name' => 'Export Users', 'group' => 'User Management'],

            // Role Management
            ['name' => 'roles.view', 'display_name' => 'View Roles', 'group' => 'Role Management'],
            ['name' => 'roles.create', 'display_name' => 'Create Roles', 'group' => 'Role Management'],
            ['name' => 'roles.update', 'display_name' => 'Update Roles', 'group' => 'Role Management'],
            ['name' => 'roles.delete', 'display_name' => 'Delete Roles', 'group' => 'Role Management'],

            // Permission Management
            ['name' => 'permissions.view', 'display_name' => 'View Permissions', 'group' => 'Permission Management'],
            ['name' => 'permissions.create', 'display_name' => 'Create Permissions', 'group' => 'Permission Management'],
            ['name' => 'permissions.update', 'display_name' => 'Update Permissions', 'group' => 'Permission Management'],
            ['name' => 'permissions.delete', 'display_name' => 'Delete Permissions', 'group' => 'Permission Management'],

            // Configuration Management
            ['name' => 'configurations.view', 'display_name' => 'View Configurations', 'group' => 'Configuration Management'],
            ['name' => 'configurations.update', 'display_name' => 'Update Configurations', 'group' => 'Configuration Management'],
            ['name' => 'configurations.backup', 'display_name' => 'Backup Configurations', 'group' => 'Configuration Management'],
            ['name' => 'configurations.restore', 'display_name' => 'Restore Configurations', 'group' => 'Configuration Management'],

            // Activity & Audit
            ['name' => 'activities.view', 'display_name' => 'View Activity Logs', 'group' => 'Audit & Activity'],
            ['name' => 'activities.export', 'display_name' => 'Export Activity Logs', 'group' => 'Audit & Activity'],
            ['name' => 'activities.purge', 'display_name' => 'Purge Activity Logs', 'group' => 'Audit & Activity'],

            // Session Management
            ['name' => 'sessions.view', 'display_name' => 'View Admin Sessions', 'group' => 'Session Management'],
            ['name' => 'sessions.terminate', 'display_name' => 'Terminate Sessions', 'group' => 'Session Management'],
            ['name' => 'sessions.cleanup', 'display_name' => 'Cleanup Old Sessions', 'group' => 'Session Management'],

            // System Settings
            ['name' => 'settings.view', 'display_name' => 'View System Settings', 'group' => 'System Settings'],
            ['name' => 'settings.update', 'display_name' => 'Update System Settings', 'group' => 'System Settings'],

            // Dashboard & Reporting
            ['name' => 'dashboard.view', 'display_name' => 'View Dashboard', 'group' => 'Dashboard & Reporting'],
            ['name' => 'dashboard.statistics', 'display_name' => 'View Statistics', 'group' => 'Dashboard & Reporting'],
        ];

        // Create permissions
        $permissionIds = [];
        foreach ($adminPermissions as $permissionData) {
            $permission = Permission::firstOrCreate(
                ['name' => $permissionData['name']],
                [
                    'display_name' => $permissionData['display_name'],
                    'group' => $permissionData['group'],
                ]
            );
            $permissionIds[] = $permission->id;
        }

        // Assign all permissions to super_admin
        $superAdmin->syncPermissions($permissionIds);

        // Assign most permissions to admin (exclude purge and cleanup)
        $adminPermissionIds = Permission::where('group', '!=', 'System Settings')
            ->whereNotIn('name', ['activities.purge', 'sessions.cleanup'])
            ->pluck('id')
            ->toArray();
        $admin->syncPermissions($adminPermissionIds);

        // Assign limited permissions to manager
        $managerPermissionIds = Permission::whereIn('name', [
            'users.view',
            'users.export',
            'activities.view',
            'sessions.view',
            'dashboard.view',
            'dashboard.statistics',
        ])->pluck('id')->toArray();
        $manager->syncPermissions($managerPermissionIds);

        $this->command->info('Admin configuration seeded successfully!');
    }
}

