<?php

namespace Database\Seeders;

use App\Models\Permission;
use App\Models\Role;
use Illuminate\Database\Seeder;

class BusinessPermissionsSeeder extends Seeder
{
    public function run()
    {
        $permissions = [
            // Business Dashboard
            ['name' => 'view-business-dashboard', 'display_name' => 'View Business Dashboard', 'group' => 'business', 'description' => 'Access business dashboard'],

            // Venue Management
            ['name' => 'manage-venues', 'display_name' => 'Manage Venues', 'group' => 'business', 'description' => 'Full venue management access'],
            ['name' => 'create-venues', 'display_name' => 'Create Venues', 'group' => 'business', 'description' => 'Create new venues'],
            ['name' => 'edit-venues', 'display_name' => 'Edit Venues', 'group' => 'business', 'description' => 'Edit venue details'],
            ['name' => 'delete-venues', 'display_name' => 'Delete Venues', 'group' => 'business', 'description' => 'Delete venues'],

            // Offer Management
            ['name' => 'manage-offers', 'display_name' => 'Manage Offers', 'group' => 'business', 'description' => 'Full offer management access'],
            ['name' => 'create-offers', 'display_name' => 'Create Offers', 'group' => 'business', 'description' => 'Create new offers'],
            ['name' => 'edit-offers', 'display_name' => 'Edit Offers', 'group' => 'business', 'description' => 'Edit offer details'],
            ['name' => 'delete-offers', 'display_name' => 'Delete Offers', 'group' => 'business', 'description' => 'Delete offers'],

            // Team Management
            ['name' => 'manage-team', 'display_name' => 'Manage Team', 'group' => 'business', 'description' => 'Manage team members'],
            ['name' => 'invite-team', 'display_name' => 'Invite Team Members', 'group' => 'business', 'description' => 'Invite new team members'],
            ['name' => 'remove-team', 'display_name' => 'Remove Team Members', 'group' => 'business', 'description' => 'Remove team members'],

            // Analytics
            ['name' => 'view-analytics', 'display_name' => 'View Analytics', 'group' => 'business', 'description' => 'View business analytics and reports'],

            // Redemptions
            ['name' => 'manage-redemptions', 'display_name' => 'Manage Redemptions', 'group' => 'business', 'description' => 'Process offer redemptions'],
            ['name' => 'view-redemptions', 'display_name' => 'View Redemptions', 'group' => 'business', 'description' => 'View redemption history'],

            // Subscription
            ['name' => 'view-subscription', 'display_name' => 'View Subscription', 'group' => 'business', 'description' => 'View subscription details'],
            ['name' => 'manage-subscription', 'display_name' => 'Manage Subscription', 'group' => 'business', 'description' => 'Manage subscription plan'],

            // Support
            ['name' => 'create-support-tickets', 'display_name' => 'Create Support Tickets', 'group' => 'business', 'description' => 'Create support tickets'],
            ['name' => 'view-support-tickets', 'display_name' => 'View Support Tickets', 'group' => 'business', 'description' => 'View support tickets'],
        ];

        // Create or update permissions
        foreach ($permissions as $permissionData) {
            Permission::updateOrCreate(
                ['name' => $permissionData['name']],
                $permissionData
            );
        }

        $this->command->info('Business permissions created successfully!');

        // Create Business Owner role if not exists
        $businessOwnerRole = Role::firstOrCreate(
            ['name' => 'business_owner'],
            [
                'display_name' => 'Business Owner',
                'description' => 'Full access to business features',
                'type' => 'business',
                'is_system' => true
            ]
        );

        // Assign all business permissions to business_owner role
        $businessPermissions = Permission::where('group', 'business')->pluck('id');
        $businessOwnerRole->permissions()->sync($businessPermissions);
        $this->command->info('All business permissions assigned to business_owner role!');

        // Create Business Manager role
        $businessManagerRole = Role::firstOrCreate(
            ['name' => 'business_manager'],
            [
                'display_name' => 'Business Manager',
                'description' => 'Manage venues, offers, and team',
                'type' => 'business',
                'is_system' => false
            ]
        );

        // Assign specific permissions to business_manager role
        $managerPermissions = Permission::whereIn('name', [
            'view-business-dashboard',
            'manage-venues',
            'create-venues',
            'edit-venues',
            'manage-offers',
            'create-offers',
            'edit-offers',
            'manage-team',
            'view-analytics',
            'manage-redemptions',
            'view-subscription',
        ])->pluck('id');

        $businessManagerRole->permissions()->sync($managerPermissions);
        $this->command->info('Business manager permissions assigned!');

        // Create Business Staff role
        $businessStaffRole = Role::firstOrCreate(
            ['name' => 'business_staff'],
            [
                'display_name' => 'Business Staff',
                'description' => 'Basic business access for staff members',
                'type' => 'business',
                'is_system' => false
            ]
        );

        // Assign limited permissions to business_staff role
        $staffPermissions = Permission::whereIn('name', [
            'view-business-dashboard',
            'manage-redemptions',
            'view-redemptions',
            'create-support-tickets',
        ])->pluck('id');

        $businessStaffRole->permissions()->sync($staffPermissions);
        $this->command->info('Business staff permissions assigned!');
    }
}
