<?php

namespace Database\Seeders;

use App\Models\Chat;
use App\Models\Message;
use App\Models\Team;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Schema;

class ChatSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('🌱 Seeding chats and messages...');

        // Check if Chat table has proper structure
        if (!Schema::hasTable('chats')) {
            $this->command->warn('⚠️ Chat system tables not properly set up. Skipping chat seeding.');
            return;
        }

        $teams = Team::with('members')->get();

        if ($teams->isEmpty()) {
            $this->command->warn('⚠️ No teams found. Skipping chat seeding.');
            return;
        }

        foreach ($teams as $team) {
            try {
                // Create team chat
                $creator = $team->members->first()?->pivot->user_id ?? User::first()?->id;
                if (!$creator) {
                    continue;
                }

                $chat = Chat::create([
                    'team_id' => $team->id,
                    'business_id' => $team->business_id,
                    'name' => $team->name . ' Chat',
                    'description' => 'Main chat channel for ' . $team->name,
                    'type' => 'team',
                    'created_by' => $creator,
                    'is_active' => true,
                    'message_count' => 0,
                ]);

                // Add all team members as participants
                $participantIds = $team->members->pluck('id')->toArray();
                foreach ($participantIds as $userId) {
                    $chat->participants()->attach($userId, [
                        'role' => 'member',
                        'joined_at' => now(),
                        'is_active' => true,
                    ]);
                }

                $this->command->info("✅ Created chat for team: {$team->name}");
            } catch (\Exception $e) {
                $this->command->warn("⚠️ Error creating chat for {$team->name}: " . $e->getMessage());
                continue;
            }
        }

        $this->command->info('✅ Chats seeded successfully!');
    }
}

