<?php

namespace Database\Seeders;

use App\Models\Role;
use App\Models\Permission;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class RolePermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Clear existing data
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        DB::table('role_has_permissions')->truncate();
        DB::table('roles')->truncate();
        DB::table('permissions')->truncate();
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        $this->command->info('🌱 Seeding roles and permissions...');

        // Define all permissions with groups
        $permissionData = [
            // User Management
            'user_management' => [
                ['name' => 'users.view', 'display_name' => 'View Users'],
                ['name' => 'users.create', 'display_name' => 'Create Users'],
                ['name' => 'users.edit', 'display_name' => 'Edit Users'],
                ['name' => 'users.delete', 'display_name' => 'Delete Users'],
                ['name' => 'users.manage_roles', 'display_name' => 'Manage User Roles'],
            ],

            // Business Management
            'business_management' => [
                ['name' => 'businesses.view', 'display_name' => 'View Businesses'],
                ['name' => 'businesses.create', 'display_name' => 'Create Businesses'],
                ['name' => 'businesses.edit', 'display_name' => 'Edit Businesses'],
                ['name' => 'businesses.delete', 'display_name' => 'Delete Businesses'],
                ['name' => 'businesses.verify', 'display_name' => 'Verify Businesses'],
                ['name' => 'businesses.approve', 'display_name' => 'Approve Businesses'],
            ],

            // Team Management
            'team_management' => [
                ['name' => 'teams.view', 'display_name' => 'View Teams'],
                ['name' => 'teams.create', 'display_name' => 'Create Teams'],
                ['name' => 'teams.edit', 'display_name' => 'Edit Teams'],
                ['name' => 'teams.delete', 'display_name' => 'Delete Teams'],
                ['name' => 'teams.manage_members', 'display_name' => 'Manage Team Members'],
            ],

            // Staff Management
            'staff_management' => [
                ['name' => 'staff.view', 'display_name' => 'View Staff'],
                ['name' => 'staff.create', 'display_name' => 'Create Staff'],
                ['name' => 'staff.edit', 'display_name' => 'Edit Staff'],
                ['name' => 'staff.delete', 'display_name' => 'Delete Staff'],
            ],

            // Venue Management
            'venue_management' => [
                ['name' => 'venues.view', 'display_name' => 'View Venues'],
                ['name' => 'venues.create', 'display_name' => 'Create Venues'],
                ['name' => 'venues.edit', 'display_name' => 'Edit Venues'],
                ['name' => 'venues.delete', 'display_name' => 'Delete Venues'],
                ['name' => 'venues.approve', 'display_name' => 'Approve Venues'],
            ],

            // Offer Management
            'offer_management' => [
                ['name' => 'offers.view', 'display_name' => 'View Offers'],
                ['name' => 'offers.create', 'display_name' => 'Create Offers'],
                ['name' => 'offers.edit', 'display_name' => 'Edit Offers'],
                ['name' => 'offers.delete', 'display_name' => 'Delete Offers'],
                ['name' => 'offers.approve', 'display_name' => 'Approve Offers'],
            ],

            // Chat Management
            'chat_management' => [
                ['name' => 'chats.view', 'display_name' => 'View Chats'],
                ['name' => 'chats.create', 'display_name' => 'Create Chats'],
                ['name' => 'chats.manage_participants', 'display_name' => 'Manage Chat Participants'],
                ['name' => 'messages.send', 'display_name' => 'Send Messages'],
                ['name' => 'messages.edit', 'display_name' => 'Edit Messages'],
                ['name' => 'messages.delete', 'display_name' => 'Delete Messages'],
            ],

            // Notification Management
            'notification_management' => [
                ['name' => 'notifications.view', 'display_name' => 'View Notifications'],
                ['name' => 'notifications.manage_templates', 'display_name' => 'Manage Notification Templates'],
                ['name' => 'notifications.send', 'display_name' => 'Send Notifications'],
            ],

            // System Management
            'system_management' => [
                ['name' => 'admin.access', 'display_name' => 'Access Admin Panel'],
                ['name' => 'admin.dashboard', 'display_name' => 'View Admin Dashboard'],
                ['name' => 'admin.settings', 'display_name' => 'Manage Admin Settings'],
                ['name' => 'roles.view', 'display_name' => 'View Roles'],
                ['name' => 'roles.manage', 'display_name' => 'Manage Roles'],
                ['name' => 'permissions.view', 'display_name' => 'View Permissions'],
                ['name' => 'permissions.manage', 'display_name' => 'Manage Permissions'],
                ['name' => 'audit_logs.view', 'display_name' => 'View Audit Logs'],
                ['name' => 'activities.view', 'display_name' => 'View Activities'],
                ['name' => 'sessions.view', 'display_name' => 'View Sessions'],
                ['name' => 'configurations.view', 'display_name' => 'View Configurations'],
                ['name' => 'settings.view', 'display_name' => 'View Settings'],
            ],

            // Spotlight Management
            'spotlight_management' => [
                ['name' => 'spotlight.view', 'display_name' => 'View Spotlights'],
                ['name' => 'spotlight.create', 'display_name' => 'Create Spotlight'],
                ['name' => 'spotlight.edit', 'display_name' => 'Edit Spotlight'],
                ['name' => 'spotlight.delete', 'display_name' => 'Delete Spotlight'],
                ['name' => 'spotlight.approve', 'display_name' => 'Approve/Reject Spotlight'],
                ['name' => 'spotlight.publish', 'display_name' => 'Publish Spotlight'],
                ['name' => 'spotlight.manage_categories', 'display_name' => 'Manage Spotlight Categories'],
                ['name' => 'spotlight.view_analytics', 'display_name' => 'View Spotlight Analytics'],
            ],

            // Redemption Management
            'redemption_management' => [
                ['name' => 'redemptions.view', 'display_name' => 'View Redemptions'],
                ['name' => 'redemptions.process', 'display_name' => 'Process Redemptions'],
                ['name' => 'redemptions.scan', 'display_name' => 'Scan QR Codes'],
            ],
        ];

        // Create permissions
        $permissionIds = [];
        foreach ($permissionData as $group => $permissions) {
            foreach ($permissions as $permission) {
                $perm = Permission::create([
                    'name' => $permission['name'],
                    'display_name' => $permission['display_name'],
                    'group' => $group,
                    'description' => null,
                    'guard_name' => 'web',
                ]);
                $permissionIds[$permission['name']] = $perm->id;
            }
        }

        $this->command->info('✅ Created ' . count($permissionIds) . ' permissions');

        // Define roles with their permissions
        $roleData = [
            'super_admin' => [
                'display_name' => 'Super Administrator',
                'description' => 'Full system access with configuration control',
                'type' => 'admin',
                'permissions' => array_keys($permissionIds), // All permissions
            ],
            'admin' => [
                'display_name' => 'Administrator',
                'description' => 'Platform management and moderation',
                'type' => 'admin',
                'permissions' => array_filter(array_keys($permissionIds), function($p) {
                    return !in_array($p, ['admin.settings', 'roles.manage', 'permissions.manage']);
                }),
            ],
            'business_owner' => [
                'display_name' => 'Business Owner',
                'description' => 'Full business management access',
                'type' => 'business',
                'permissions' => [
                    'businesses.view', 'businesses.edit',
                    'teams.view', 'teams.create', 'teams.edit', 'teams.manage_members',
                    'staff.view', 'staff.create', 'staff.edit',
                    'venues.view', 'venues.create', 'venues.edit',
                    'offers.view', 'offers.create', 'offers.edit',
                    'chats.view', 'chats.create', 'chats.manage_participants',
                    'messages.send', 'messages.edit',
                    'notifications.view',
                ],
            ],
            'business_manager' => [
                'display_name' => 'Business Manager',
                'description' => 'Manage venues and offers',
                'type' => 'business',
                'permissions' => [
                    'businesses.view',
                    'teams.view', 'teams.manage_members',
                    'staff.view', 'staff.create',
                    'venues.view', 'venues.create', 'venues.edit',
                    'offers.view', 'offers.create', 'offers.edit',
                    'chats.view', 'messages.send',
                    'notifications.view',
                ],
            ],
            'venue_staff' => [
                'display_name' => 'Venue Staff',
                'description' => 'Scan and redeem offers',
                'type' => 'business',
                'permissions' => [
                    'redemptions.view', 'redemptions.process', 'redemptions.scan',
                    'messages.send',
                    'chats.view',
                ],
            ],
            'user' => [
                'display_name' => 'User',
                'description' => 'Regular app user',
                'type' => 'user',
                'permissions' => [],
            ],
        ];

        // Create roles and assign permissions
        foreach ($roleData as $roleName => $data) {
            $role = Role::create([
                'name' => $roleName,
                'display_name' => $data['display_name'],
                'description' => $data['description'],
                'type' => $data['type'],
                'is_system' => true,
                'guard_name' => 'web',
            ]);

            // Assign permissions to role
            $permIds = [];
            foreach ($data['permissions'] as $permName) {
                if (isset($permissionIds[$permName])) {
                    $permIds[] = $permissionIds[$permName];
                }
            }

            if (!empty($permIds)) {
                $role->permissions()->sync($permIds);
            }

            $this->command->info("✅ Created role: {$role->display_name}");
        }

        $this->command->newLine();
        $this->command->info('✅ Roles and permissions seeded successfully!');
    }
}

