<?php

namespace Database\Seeders;

use App\Models\Business;
use App\Models\Role;
use App\Models\Team;
use App\Models\User;
use Illuminate\Database\Seeder;

class TeamSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('🌱 Seeding teams...');

        // Get or create admin user
        $admin = User::where('email', 'admin@privasee.com')->first();
        if (!$admin) {
            $admin = User::factory()->create([
                'email' => 'admin@privasee.com',
                'first_name' => 'Admin',
                'last_name' => 'User',
                'account_type' => 'admin',
            ]);
        }

        // Get business or create one directly
        $business = Business::first();
        if (!$business) {
            $business = Business::create([
                'business_name' => 'Test Business',
                'business_name_ar' => 'عمل تجريبي',
                'owner_id' => $admin->id,
                'email' => 'business@example.com',
                'phone' => '+971501234567',
                'trade_license_number' => 'TLN' . time(),
                'trade_license_document' => 'test-license.pdf',
                'description' => 'A test business for seeding',
                'status' => 'verified',
                'verified_at' => now(),
            ]);
        }

        $this->command->info("Creating teams for business: {$business->business_name}");

        // Get roles
        $managerRole = Role::where('name', 'business_manager')->first();
        $staffRole = Role::where('name', 'venue_staff')->first();

        // Define teams
        $teams = [
            [
                'name' => 'Management Team',
                'description' => 'Team for business management and strategic decisions',
                'status' => 'active',
            ],
            [
                'name' => 'Operations Team',
                'description' => 'Team for daily operations and staff management',
                'status' => 'active',
            ],
            [
                'name' => 'Marketing Team',
                'description' => 'Team for marketing campaigns and promotions',
                'status' => 'active',
            ],
            [
                'name' => 'Support Team',
                'description' => 'Team for customer support and inquiries',
                'status' => 'active',
            ],
        ];

        foreach ($teams as $teamData) {
            $team = Team::create([
                'business_id' => $business->id,
                'name' => $teamData['name'],
                'description' => $teamData['description'],
                'status' => $teamData['status'],
                'member_count' => 0,
            ]);

            // Add admin as team manager
            $team->members()->attach($admin->id, [
                'role_id' => $managerRole?->id,
                'custom_permissions' => json_encode([]),
                'status' => 'active',
                'joined_at' => now(),
            ]);

            // Create and add staff members to team
            for ($i = 0; $i < 3; $i++) {
                $user = User::factory()->create([
                    'account_type' => 'business',
                ]);

                $team->members()->attach($user->id, [
                    'role_id' => $staffRole?->id,
                    'custom_permissions' => json_encode([]),
                    'status' => 'active',
                    'joined_at' => now(),
                ]);
            }

            // Update member count
            $memberCount = $team->members()->where('team_user.status', 'active')->count();
            $team->update(['member_count' => $memberCount]);

            $this->command->info("✅ Created team: {$team->name} with {$memberCount} members");
        }

        $this->command->info('✅ Teams seeded successfully!');
    }
}

