<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\Role;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;

class UserSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $this->command->info('🌱 Seeding users...');

        // Get roles
        $superAdminRole = Role::where('name', 'super_admin')->first();
        $adminRole = Role::where('name', 'admin')->first();
        $businessOwnerRole = Role::where('name', 'business_owner')->first();
        $userRole = Role::where('name', 'user')->first();

        // Test Users
        $users = [
            [
                'first_name' => 'Super',
                'last_name' => 'Admin',
                'email' => 'superadmin@privasee.com',
                'phone' => '+971501234567',
                'password' => 'password123',
                'account_type' => 'admin',
                'status' => 'active',
                'role' => $superAdminRole,
            ],
            [
                'first_name' => 'Admin',
                'last_name' => 'User',
                'email' => 'admin@privasee.com',
                'phone' => '+971501234568',
                'password' => 'password123',
                'account_type' => 'admin',
                'status' => 'active',
                'role' => $adminRole,
            ],
            [
                'first_name' => 'Business',
                'last_name' => 'Owner',
                'email' => 'owner@business.com',
                'phone' => '+971501234569',
                'password' => 'password123',
                'account_type' => 'business',
                'status' => 'active',
                'role' => $businessOwnerRole,
            ],
            [
                'first_name' => 'John',
                'last_name' => 'Doe',
                'email' => 'john@privasee.com',
                'phone' => '+971501234570',
                'password' => 'password123',
                'account_type' => 'user',
                'status' => 'active',
                'role' => $userRole,
            ],
            [
                'first_name' => 'Jane',
                'last_name' => 'Smith',
                'email' => 'jane@privasee.com',
                'phone' => '+971501234571',
                'password' => 'password123',
                'account_type' => 'user',
                'status' => 'active',
                'role' => $userRole,
            ],
        ];

        foreach ($users as $userData) {
            $role = $userData['role'];
            unset($userData['role']);

            $user = User::updateOrCreate(
                ['email' => $userData['email']],
                [
                    'password' => Hash::make($userData['password']),
                    'first_name' => $userData['first_name'],
                    'last_name' => $userData['last_name'],
                    'phone' => $userData['phone'],
                    'account_type' => $userData['account_type'],
                    'status' => $userData['status'],
                    'email_verified_at' => now(),
                    'phone_verified_at' => now(),
                ]
            );

            // Assign role
            if ($role) {
                $user->roles()->syncWithoutDetaching([$role->id]);
            }

            $this->command->info("✅ Created user: {$user->first_name} {$user->last_name} ({$user->email})");
        }

        // Create additional test users
        $this->command->info('Creating additional test users...');
        for ($i = 1; $i <= 5; $i++) {
            $user = User::factory()->create([
                'account_type' => 'business',
                'status' => 'active',
                'email_verified_at' => now(),
                'phone_verified_at' => now(),
            ]);

            if ($userRole) {
                $user->roles()->attach($userRole->id);
            }

            $this->command->info("✅ Created test user: {$user->first_name} {$user->last_name}");
        }

        $this->command->info('✅ Users seeded successfully!');
    }
}

