@extends('layouts.admin')

@section('title', 'Marketing Campaigns Management')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-md-8">
            <h1 class="h2 fw-bold mb-0">
                <i class="fas fa-bullhorn me-2"></i>Marketing Campaigns
            </h1>
            <p class="text-muted">Manage all marketing campaigns across the platform</p>
        </div>
        <div class="col-md-4 text-end">
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#createCampaignModal">
                <i class="fas fa-plus me-2"></i>New Campaign
            </button>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card shadow-sm border-0">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted small mb-1">Total Campaigns</p>
                            <h3 class="mb-0">{{ $totalCampaigns }}</h3>
                        </div>
                        <div class="bg-primary bg-opacity-10 rounded-circle p-3">
                            <i class="fas fa-campaign text-primary fs-5"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm border-0">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted small mb-1">Active</p>
                            <h3 class="mb-0">{{ $activeCampaigns }}</h3>
                        </div>
                        <div class="bg-success bg-opacity-10 rounded-circle p-3">
                            <i class="fas fa-play-circle text-success fs-5"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm border-0">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted small mb-1">Total Impressions</p>
                            <h3 class="mb-0">{{ number_format($totalImpressions) }}</h3>
                        </div>
                        <div class="bg-info bg-opacity-10 rounded-circle p-3">
                            <i class="fas fa-eye text-info fs-5"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card shadow-sm border-0">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <div>
                            <p class="text-muted small mb-1">Total Clicks</p>
                            <h3 class="mb-0">{{ number_format($totalClicks) }}</h3>
                        </div>
                        <div class="bg-warning bg-opacity-10 rounded-circle p-3">
                            <i class="fas fa-mouse text-warning fs-5"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="card shadow-sm border-0 mb-4">
        <div class="card-body">
            <div class="row g-3">
                <div class="col-md-4">
                    <label class="form-label small fw-bold">Search Campaigns</label>
                    <input type="text" class="form-control" id="searchInput" placeholder="Campaign name, business...">
                </div>
                <div class="col-md-3">
                    <label class="form-label small fw-bold">Status</label>
                    <select class="form-select" id="statusFilter">
                        <option value="">All Status</option>
                        <option value="active">Active</option>
                        <option value="paused">Paused</option>
                        <option value="completed">Completed</option>
                        <option value="rejected">Rejected</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label small fw-bold">Type</label>
                    <select class="form-select" id="typeFilter">
                        <option value="">All Types</option>
                        <option value="banner">Banner</option>
                        <option value="social">Social Media</option>
                        <option value="email">Email</option>
                    </select>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button class="btn btn-outline-secondary w-100" onclick="resetFilters()">
                        <i class="fas fa-redo me-1"></i>Reset
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Campaigns Table -->
    <div class="card shadow-sm border-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light border-bottom">
                    <tr>
                        <th>Campaign Name</th>
                        <th>Business</th>
                        <th>Type</th>
                        <th>Status</th>
                        <th>Duration</th>
                        <th>Budget</th>
                        <th>Impressions</th>
                        <th>Clicks</th>
                        <th>CTR</th>
                        <th class="text-end pe-4">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($campaigns as $campaign)
                    <tr>
                        <td>
                            <div>
                                <p class="fw-bold mb-0">{{ $campaign->name }}</p>
                                <small class="text-muted">{{ Str::limit($campaign->description, 40) }}</small>
                            </div>
                        </td>
                        <td>
                            <small>{{ $campaign->business->business_name ?? 'N/A' }}</small>
                        </td>
                        <td>
                            <span class="badge bg-secondary">{{ ucfirst($campaign->type) }}</span>
                        </td>
                        <td>
                            @if($campaign->status === 'active')
                            <span class="badge bg-success">Active</span>
                            @elseif($campaign->status === 'paused')
                            <span class="badge bg-warning">Paused</span>
                            @elseif($campaign->status === 'completed')
                            <span class="badge bg-info">Completed</span>
                            @else
                            <span class="badge bg-danger">Rejected</span>
                            @endif
                        </td>
                        <td>
                            <small>{{ $campaign->start_date->format('M d') }} - {{ $campaign->end_date->format('M d') }}</small>
                        </td>
                        <td>
                            <span class="fw-bold">{{ $campaign->currency }} {{ number_format($campaign->budget, 2) }}</span>
                        </td>
                        <td>
                            <small>{{ number_format($campaign->metrics['impressions'] ?? 0) }}</small>
                        </td>
                        <td>
                            <small>{{ number_format($campaign->metrics['clicks'] ?? 0) }}</small>
                        </td>
                        <td>
                            <small>
                                @php
                                $impressions = $campaign->metrics['impressions'] ?? 1;
                                $clicks = $campaign->metrics['clicks'] ?? 0;
                                $ctr = $impressions > 0 ? (($clicks / $impressions) * 100) : 0;
                                @endphp
                                {{ number_format($ctr, 2) }}%
                            </small>
                        </td>
                        <td class="text-end pe-4">
                            <div class="btn-group btn-group-sm">
                                <a href="{{ route('admin.marketing.campaigns.show', $campaign->id) }}" class="btn btn-outline-primary" title="View">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <button class="btn btn-outline-secondary" onclick="editCampaign({{ $campaign->id }})" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </button>
                                @if($campaign->status === 'active')
                                <button class="btn btn-outline-warning btn-sm" onclick="pauseCampaign({{ $campaign->id }})" title="Pause">
                                    <i class="fas fa-pause"></i>
                                </button>
                                @elseif($campaign->status === 'paused')
                                <button class="btn btn-outline-success btn-sm" onclick="resumeCampaign({{ $campaign->id }})" title="Resume">
                                    <i class="fas fa-play"></i>
                                </button>
                                @endif
                                <button class="btn btn-outline-danger" onclick="deleteCampaign({{ $campaign->id }})" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="10" class="text-center py-5">
                            <i class="fas fa-inbox text-muted" style="font-size: 48px;"></i>
                            <p class="text-muted mt-3">No campaigns found</p>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        <div class="card-footer bg-light">
            <div class="d-flex justify-content-between align-items-center">
                <p class="text-muted small mb-0">Showing {{ $campaigns->count() }} of {{ $totalCampaigns }} campaigns</p>
                {{ $campaigns->links() }}
            </div>
        </div>
    </div>
</div>

<!-- Edit Campaign Modal -->
<div class="modal fade" id="editCampaignModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title fw-bold">Edit Campaign</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form id="editCampaignForm">
                @csrf
                <input type="hidden" id="editCampaignId" name="id">
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Campaign Name</label>
                            <input type="text" class="form-control" id="editCampaignName" name="name" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Campaign Type</label>
                            <select class="form-select" id="editCampaignType" name="type">
                                <option value="banner">Banner</option>
                                <option value="social">Social Media</option>
                                <option value="email">Email</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea class="form-control" id="editCampaignDesc" name="description" rows="3"></textarea>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Start Date</label>
                            <input type="datetime-local" class="form-control" id="editStartDate" name="start_date">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">End Date</label>
                            <input type="datetime-local" class="form-control" id="editEndDate" name="end_date">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Budget</label>
                            <input type="number" class="form-control" id="editBudget" name="budget" step="0.01">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Status</label>
                            <select class="form-select" id="editCampaignStatus" name="status">
                                <option value="active">Active</option>
                                <option value="paused">Paused</option>
                                <option value="completed">Completed</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="modal-footer bg-light">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function editCampaign(id) {
    fetch(`/admin/marketing/campaigns/${id}`)
        .then(r => r.json())
        .then(data => {
            document.getElementById('editCampaignId').value = data.campaign.id;
            document.getElementById('editCampaignName').value = data.campaign.name;
            document.getElementById('editCampaignType').value = data.campaign.type;
            document.getElementById('editCampaignDesc').value = data.campaign.description || '';
            document.getElementById('editStartDate').value = data.campaign.start_date;
            document.getElementById('editEndDate').value = data.campaign.end_date;
            document.getElementById('editBudget').value = data.campaign.budget;
            document.getElementById('editCampaignStatus').value = data.campaign.status;
            new bootstrap.Modal(document.getElementById('editCampaignModal')).show();
        });
}

function pauseCampaign(id) {
    if(confirm('Pause this campaign?')) {
        fetch(`/admin/marketing/campaigns/${id}/pause`, {method: 'POST', headers: {'X-CSRF-TOKEN': '{{ csrf_token() }}'}})
            .then(r => r.json())
            .then(d => {if(d.success) location.reload();});
    }
}

function resumeCampaign(id) {
    if(confirm('Resume this campaign?')) {
        fetch(`/admin/marketing/campaigns/${id}/resume`, {method: 'POST', headers: {'X-CSRF-TOKEN': '{{ csrf_token() }}'}})
            .then(r => r.json())
            .then(d => {if(d.success) location.reload();});
    }
}

function deleteCampaign(id) {
    if(confirm('Delete this campaign? This action cannot be undone.')) {
        fetch(`/admin/marketing/campaigns/${id}`, {method: 'DELETE', headers: {'X-CSRF-TOKEN': '{{ csrf_token() }}'}})
            .then(r => r.json())
            .then(d => {if(d.success) location.reload();});
    }
}

function resetFilters() {
    document.getElementById('searchInput').value = '';
    document.getElementById('statusFilter').value = '';
    document.getElementById('typeFilter').value = '';
}
</script>
@endsection

