@extends('layouts.admin')

@section('title', 'Marketing Campaigns')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-md-8">
            <h1 class="h2 fw-bold mb-0">
                <i class="fas fa-megaphone me-2"></i>Marketing Campaigns
            </h1>
            <p class="text-muted">Manage business marketing campaigns and promotions</p>
        </div>
        <div class="col-md-4 text-end">
            <a href="{{ route('admin.marketing.campaigns.create') }}" class="btn btn-primary">
                <i class="fas fa-plus me-2"></i>Create Campaign
            </a>
        </div>
    </div>

    <!-- Statistics -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h3 class="mb-0">{{ $stats['total'] ?? 0 }}</h3>
                    <p class="mb-0 small">Total Campaigns</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h3 class="mb-0">{{ $stats['approved'] ?? 0 }}</h3>
                    <p class="mb-0 small">Approved</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-warning text-dark">
                <div class="card-body">
                    <h3 class="mb-0">{{ $stats['pending'] ?? 0 }}</h3>
                    <p class="mb-0 small">Pending Review</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-info text-white">
                <div class="card-body">
                    <h3 class="mb-0">{{ $stats['active'] ?? 0 }}</h3>
                    <p class="mb-0 small">Active Now</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <input type="text" name="search" class="form-control" placeholder="Search campaigns..."
                           value="{{ request('search') }}">
                </div>
                <div class="col-md-3">
                    <select name="status" class="form-select">
                        <option value="">All Status</option>
                        <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="approved" {{ request('status') === 'approved' ? 'selected' : '' }}>Approved</option>
                        <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active</option>
                        <option value="completed" {{ request('status') === 'completed' ? 'selected' : '' }}>Completed</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <select name="type" class="form-select">
                        <option value="">All Types</option>
                        <option value="banner" {{ request('type') === 'banner' ? 'selected' : '' }}>Banner</option>
                        <option value="spotlight" {{ request('type') === 'spotlight' ? 'selected' : '' }}>Spotlight</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">Filter</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Campaigns Table -->
    <div class="card">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Campaign</th>
                        <th>Business</th>
                        <th>Type</th>
                        <th>Status</th>
                        <th>Period</th>
                        <th>Stats</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($campaigns as $campaign)
                    <tr>
                        <td>
                            <strong>{{ $campaign->name }}</strong>
                            <br>
                            <small class="text-muted">{{ Str::limit($campaign->description, 50) }}</small>
                        </td>
                        <td>
                            <strong>{{ $campaign->business->business_name ?? 'N/A' }}</strong>
                            <br>
                            <small class="text-muted">{{ $campaign->business->email ?? '' }}</small>
                        </td>
                        <td>
                            <span class="badge bg-secondary">{{ ucfirst($campaign->type) }}</span>
                        </td>
                        <td>
                            @if($campaign->status === 'pending')
                                <span class="badge bg-warning text-dark">Pending Review</span>
                            @elseif($campaign->status === 'approved')
                                <span class="badge bg-success">Approved</span>
                            @elseif($campaign->status === 'active')
                                <span class="badge bg-info">Active</span>
                            @elseif($campaign->status === 'completed')
                                <span class="badge bg-secondary">Completed</span>
                            @else
                                <span class="badge bg-danger">{{ ucfirst($campaign->status) }}</span>
                            @endif
                        </td>
                        <td>
                            {{ $campaign->start_date->format('M d') }} - {{ $campaign->end_date->format('M d, Y') }}
                        </td>
                        <td>
                            <small>
                                <i class="fas fa-eye me-1"></i> {{ $campaign->impressions ?? 0 }}<br>
                                <i class="fas fa-mouse me-1"></i> {{ $campaign->clicks ?? 0 }}<br>
                                <i class="fas fa-check me-1"></i> {{ $campaign->conversions ?? 0 }}
                            </small>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                <a href="#" class="btn btn-info" title="View" data-bs-toggle="modal"
                                   data-bs-target="#viewModal" onclick="showCampaign({{ $campaign->id }})">
                                    <i class="fas fa-eye"></i>
                                </a>
                                @if($campaign->status === 'pending')
                                <button class="btn btn-success" title="Approve" onclick="approveCampaign({{ $campaign->id }})">
                                    <i class="fas fa-check"></i>
                                </button>
                                <button class="btn btn-danger" title="Reject" onclick="rejectCampaign({{ $campaign->id }})">
                                    <i class="fas fa-times"></i>
                                </button>
                                @elseif($campaign->status === 'approved' || $campaign->status === 'active')
                                <button class="btn btn-warning" title="Deactivate" onclick="deactivateCampaign({{ $campaign->id }})">
                                    <i class="fas fa-pause"></i>
                                </button>
                                @endif
                                <button class="btn btn-secondary" title="Edit" onclick="editCampaign({{ $campaign->id }})">
                                    <i class="fas fa-edit"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="7" class="text-center py-4 text-muted">
                            <i class="fas fa-inbox" style="font-size: 2rem;"></i>
                            <p class="mt-2">No campaigns found</p>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        @if($campaigns->hasPages())
        <div class="card-footer">
            {{ $campaigns->links() }}
        </div>
        @endif
    </div>
</div>

<!-- View Campaign Modal -->
<div class="modal fade" id="viewModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header bg-light">
                <h5 class="modal-title fw-bold">Campaign Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="campaignDetails">
                <div class="spinner-border" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function showCampaign(campaignId) {
    fetch(`/admin/marketing/campaigns/${campaignId}`)
        .then(response => response.json())
        .then(data => {
            let html = `
                <div class="row mb-3">
                    <div class="col-md-6">
                        <p class="text-muted small mb-1">Campaign Name</p>
                        <h6 class="fw-bold">${data.name}</h6>
                    </div>
                    <div class="col-md-6">
                        <p class="text-muted small mb-1">Business</p>
                        <h6 class="fw-bold">${data.business_name}</h6>
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <p class="text-muted small mb-1">Status</p>
                        <h6 class="fw-bold"><span class="badge bg-success">${data.status}</span></h6>
                    </div>
                    <div class="col-md-6">
                        <p class="text-muted small mb-1">Type</p>
                        <h6 class="fw-bold">${data.type}</h6>
                    </div>
                </div>
                <div class="mb-3">
                    <p class="text-muted small mb-1">Description</p>
                    <p>${data.description}</p>
                </div>
                <div class="row mb-3">
                    <div class="col-md-4 text-center">
                        <p class="text-muted small">Impressions</p>
                        <h5 class="fw-bold">${data.impressions || 0}</h5>
                    </div>
                    <div class="col-md-4 text-center">
                        <p class="text-muted small">Clicks</p>
                        <h5 class="fw-bold">${data.clicks || 0}</h5>
                    </div>
                    <div class="col-md-4 text-center">
                        <p class="text-muted small">Conversions</p>
                        <h5 class="fw-bold">${data.conversions || 0}</h5>
                    </div>
                </div>
            `;
            document.getElementById('campaignDetails').innerHTML = html;
        })
        .catch(error => {
            document.getElementById('campaignDetails').innerHTML = '<div class="alert alert-danger">Error loading campaign details</div>';
        });
}

function approveCampaign(campaignId) {
    if (confirm('Are you sure you want to approve this campaign?')) {
        fetch(`/admin/marketing/campaigns/${campaignId}/approve`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Campaign approved successfully');
                location.reload();
            }
        });
    }
}

function rejectCampaign(campaignId) {
    let reason = prompt('Please provide a reason for rejection:');
    if (reason) {
        fetch(`/admin/marketing/campaigns/${campaignId}/reject`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ reason: reason })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Campaign rejected');
                location.reload();
            }
        });
    }
}

function deactivateCampaign(campaignId) {
    if (confirm('Are you sure you want to deactivate this campaign?')) {
        fetch(`/admin/marketing/campaigns/${campaignId}/deactivate`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Campaign deactivated');
                location.reload();
            }
        });
    }
}

function editCampaign(campaignId) {
    window.location.href = `/admin/marketing/campaigns/${campaignId}/edit`;
}
</script>
@endsection

