@extends('layouts.admin')

@section('title', 'Admin Sessions')

@section('content')
<div class="container-fluid py-4">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-md-8">
            <h1 class="h3 mb-0">Admin Sessions</h1>
            <p class="text-muted">Manage and monitor admin user sessions</p>
        </div>
        <div class="col-md-4 text-end">
            <button class="btn btn-danger btn-sm" id="cleanup-sessions">
                <i class="fas fa-trash"></i> Cleanup Old Sessions
            </button>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-title text-muted mb-2">Active Sessions</h6>
                    <h3 class="mb-0">{{ $stats['total_active'] ?? 0 }}</h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-title text-muted mb-2">Today Logins</h6>
                    <h3 class="mb-0">{{ $stats['today_logins'] ?? 0 }}</h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-title text-muted mb-2">This Week</h6>
                    <h3 class="mb-0">{{ $stats['this_week_logins'] ?? 0 }}</h3>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h6 class="card-title text-muted mb-2">Concurrent Users</h6>
                    <h3 class="mb-0">{{ $stats['concurrent_sessions'] ?? 0 }}</h3>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="get" class="row g-3">
                <div class="col-md-3">
                    <label class="form-label">Admin User</label>
                    <select name="user_id" class="form-select">
                        <option value="">All Users</option>
                        @if(isset($admins))
                        @foreach($admins as $admin)
                            <option value="{{ $admin->id }}" {{ request('user_id') == $admin->id ? 'selected' : '' }}>
                                {{ $admin->first_name ?? '' }} {{ $admin->last_name ?? '' }}
                            </option>
                        @endforeach
                        @endif
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                        <option value="">All</option>
                        <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                        <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Search (IP/Email)</label>
                    <input type="text" name="search" class="form-control" placeholder="IP address or email" value="{{ request('search') }}">
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-search"></i> Filter
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Sessions Table -->
    <div class="card">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Admin User</th>
                        <th>IP Address</th>
                        <th>Browser</th>
                        <th>Login Time</th>
                        <th>Last Activity</th>
                        <th>Status</th>
                        <th>Duration</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @if(isset($sessions))
                    @forelse($sessions as $session)
                        <tr>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="avatar-sm me-2">
                                        <span class="avatar-title rounded-circle bg-primary text-white">
                                            {{ strtoupper(substr($session->user->first_name ?? 'A', 0, 1)) }}
                                        </span>
                                    </div>
                                    <div>
                                        <h6 class="mb-0">{{ $session->user->first_name ?? 'N/A' }} {{ $session->user->last_name ?? '' }}</h6>
                                        <small class="text-muted">{{ $session->user->email ?? 'N/A' }}</small>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <code>{{ $session->ip_address ?? 'N/A' }}</code>
                            </td>
                            <td>
                                @if($session->metadata && isset($session->metadata['browser']))
                                    {{ $session->metadata['browser'] }}
                                    @if(isset($session->metadata['device']))
                                        <br><small class="text-muted">{{ $session->metadata['device'] }}</small>
                                    @endif
                                @else
                                    <span class="text-muted">Unknown</span>
                                @endif
                            </td>
                            <td>
                                <small>{{ $session->login_at ? $session->login_at->format('M d, Y H:i') : 'N/A' }}</small>
                            </td>
                            <td>
                                <small>{{ $session->last_activity_at ? $session->last_activity_at->format('M d, Y H:i') : 'N/A' }}</small>
                            </td>
                            <td>
                                @if($session->is_active && !$session->logout_at)
                                    <span class="badge bg-success">Active</span>
                                @else
                                    <span class="badge bg-secondary">Inactive</span>
                                @endif
                            </td>
                            <td>
                                <small>{{ method_exists($session, 'getDuration') ? $session->getDuration() : '0' }} seconds</small>
                            </td>
                            <td>
                                <button class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#sessionDetailModal"
                                    onclick="loadSessionDetail({{ $session->id }})">
                                    <i class="fas fa-eye"></i>
                                </button>
                                @if($session->is_active && $session->user_id !== auth()->id())
                                    <button class="btn btn-sm btn-danger" onclick="terminateSession({{ $session->id }})">
                                        <i class="fas fa-times"></i>
                                    </button>
                                @endif
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="8" class="text-center text-muted py-4">
                                No sessions found
                            </td>
                        </tr>
                    @endforelse
                    @else
                    <tr>
                        <td colspan="8" class="text-center text-danger py-4">
                            <i class="fas fa-exclamation-triangle me-2"></i>Unable to load sessions
                        </td>
                    </tr>
                    @endif
                </tbody>
            </table>
        </div>
    </div>

    <!-- Pagination -->
    <div class="d-flex justify-content-between align-items-center mt-4">
        <div>
            @if(isset($sessions))
            Showing {{ $sessions->firstItem() ?? 0 }} to {{ $sessions->lastItem() ?? 0 }} of {{ $sessions->total() }} sessions
            @else
            No data available
            @endif
        </div>
        <div>
            @if(isset($sessions) && $sessions->hasPages())
            {{ $sessions->links() }}
            @endif
        </div>
    </div>
</div>

<!-- Session Detail Modal -->
<div class="modal fade" id="sessionDetailModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Session Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div id="sessionDetailContent">
                    Loading...
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
function loadSessionDetail(sessionId) {
    // Fetch session detail
    fetch(`/api/admin/sessions/${sessionId}`)
        .then(response => response.json())
        .then(data => {
            // Display session details
            console.log(data);
        });
}

function terminateSession(sessionId) {
    if (!confirm('Are you sure you want to terminate this session?')) return;

    fetch(`/api/admin/sessions/${sessionId}/terminate`, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Session terminated successfully');
            location.reload();
        }
    });
}

document.getElementById('cleanup-sessions')?.addEventListener('click', function() {
    if (!confirm('Delete sessions older than 90 days?')) return;

    fetch('/api/admin/sessions/cleanup', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        alert(data.message);
        location.reload();
    });
});
</script>
@endpush
@endsection

