@extends('layouts.admin')

@section('title', 'Business Subscriptions')

@section('content')
<div class="container-fluid">
    <!-- Header -->
    <div class="row mb-4">
        <div class="col-md-8">
            <h1 class="h3 mb-0">Business Subscriptions</h1>
            <p class="text-muted">Manage business subscription plans and monitor usage</p>
        </div>
    </div>

    <!-- Statistics -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h3 class="mb-0">{{ $stats['active'] ?? 0 }}</h3>
                    <p class="text-muted mb-0">Active Subscriptions</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h3 class="mb-0 text-warning">{{ $stats['expiring_soon'] ?? 0 }}</h3>
                    <p class="text-muted mb-0">Expiring Soon (7 days)</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h3 class="mb-0 text-info">{{ $stats['trial'] ?? 0 }}</h3>
                    <p class="text-muted mb-0">In Trial</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card">
                <div class="card-body">
                    <h3 class="mb-0">AED {{ number_format($stats['monthly_revenue'] ?? 0, 2) }}</h3>
                    <p class="text-muted mb-0">Monthly Revenue</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <input type="text" name="search" class="form-control" placeholder="Search business..."
                           value="{{ request('search') }}">
                </div>
                <div class="col-md-3">
                    <select name="status" class="form-select">
                        <option value="">All Status</option>
                        <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>Active</option>
                        <option value="trial" {{ request('status') === 'trial' ? 'selected' : '' }}>Trial</option>
                        <option value="cancelled" {{ request('status') === 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <select name="tier" class="form-select">
                        <option value="">All Tiers</option>
                        <option value="basic" {{ request('tier') === 'basic' ? 'selected' : '' }}>Basic</option>
                        <option value="professional" {{ request('tier') === 'professional' ? 'selected' : '' }}>Professional</option>
                        <option value="enterprise" {{ request('tier') === 'enterprise' ? 'selected' : '' }}>Enterprise</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">Filter</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Subscriptions Table -->
    <div class="card">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Business</th>
                        <th>Plan</th>
                        <th>Status</th>
                        <th>Start Date</th>
                        <th>End Date</th>
                        <th>Days Left</th>
                        <th>Usage</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($subscriptions as $subscription)
                    <tr>
                        <td>
                            <strong>{{ $subscription->business->business_name }}</strong>
                            <br>
                            <small class="text-muted">{{ $subscription->business->email }}</small>
                        </td>
                        <td>
                            <span class="badge bg-secondary">{{ $subscription->plan->name }}</span>
                            <br>
                            <small>{{ $subscription->plan->tier ? ucfirst($subscription->plan->tier) : 'N/A' }}</small>
                        </td>
                        <td>
                            <span class="badge bg-{{ $subscription->status === 'active' ? 'success' : ($subscription->status === 'trial' ? 'info' : 'danger') }}">
                                {{ ucfirst($subscription->status) }}
                            </span>
                        </td>
                        <td>{{ $subscription->start_date->format('M d, Y') }}</td>
                        <td>{{ $subscription->end_date->format('M d, Y') }}</td>
                        <td>
                            @if($subscription->isActive())
                                <span class="badge bg-{{ $subscription->daysRemaining() <= 7 ? 'warning' : 'info' }}">
                                    {{ $subscription->daysRemaining() }} days
                                </span>
                            @else
                                <span class="badge bg-danger">Expired</span>
                            @endif
                        </td>
                        <td>
                            <button class="btn btn-sm btn-outline-secondary" data-bs-toggle="modal"
                                    data-bs-target="#usageModal" onclick="showUsage({{ $subscription->business_id }})">
                                View
                            </button>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm" role="group">
                                <a href="{{ route('admin.subscriptions.edit', $subscription->id) }}"
                                   class="btn btn-info" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <button class="btn btn-danger" onclick="cancelSubscription({{ $subscription->id }})"
                                        title="Cancel">
                                    <i class="fas fa-times"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="8" class="text-center py-4 text-muted">
                            <i class="fas fa-inbox" style="font-size: 2rem;"></i>
                            <p class="mt-2">No subscriptions found</p>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        @if($subscriptions->hasPages())
        <div class="card-footer">
            {{ $subscriptions->links() }}
        </div>
        @endif
    </div>
</div>

<!-- Usage Modal -->
<div class="modal fade" id="usageModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Subscription Usage</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="usageContent">
                <div class="spinner-border" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function showUsage(businessId) {
    fetch(`/admin/subscriptions/${businessId}/usage`)
        .then(response => response.json())
        .then(data => {
            let html = '<div class="row">';

            if (data.usage.venues) {
                html += `
                    <div class="col-md-6 mb-3">
                        <h6>Venues</h6>
                        <div class="progress">
                            <div class="progress-bar" style="width: ${(data.usage.venues.used / data.usage.venues.limit * 100)}%">
                                ${data.usage.venues.used} / ${data.usage.venues.limit}
                            </div>
                        </div>
                    </div>
                `;
            }

            if (data.usage.staff) {
                html += `
                    <div class="col-md-6 mb-3">
                        <h6>Staff Members</h6>
                        <div class="progress">
                            <div class="progress-bar" style="width: ${(data.usage.staff.used / data.usage.staff.limit * 100)}%">
                                ${data.usage.staff.used} / ${data.usage.staff.limit}
                            </div>
                        </div>
                    </div>
                `;
            }

            if (data.usage.offers) {
                html += `
                    <div class="col-md-6 mb-3">
                        <h6>Offers</h6>
                        <div class="progress">
                            <div class="progress-bar" style="width: ${(data.usage.offers.used / data.usage.offers.limit * 100)}%">
                                ${data.usage.offers.used} / ${data.usage.offers.limit}
                            </div>
                        </div>
                    </div>
                `;
            }

            if (data.usage.campaigns_this_month) {
                html += `
                    <div class="col-md-6 mb-3">
                        <h6>Campaigns (This Month)</h6>
                        <div class="progress">
                            <div class="progress-bar" style="width: ${(data.usage.campaigns_this_month.used / data.usage.campaigns_this_month.limit * 100)}%">
                                ${data.usage.campaigns_this_month.used} / ${data.usage.campaigns_this_month.limit}
                            </div>
                        </div>
                    </div>
                `;
            }

            html += '</div>';
            document.getElementById('usageContent').innerHTML = html;
        })
        .catch(error => {
            document.getElementById('usageContent').innerHTML = '<div class="alert alert-danger">Error loading usage data</div>';
        });
}

function cancelSubscription(subscriptionId) {
    if (confirm('Are you sure you want to cancel this subscription?')) {
        fetch(`/admin/subscriptions/${subscriptionId}/cancel`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Subscription cancelled successfully');
                location.reload();
            } else {
                alert('Error: ' + (data.message || 'Failed to cancel subscription'));
            }
        });
    }
}
</script>
@endsection

