@extends('layouts.business')

@section('content')
<div class="container mt-4">
    <div class="row mb-4">
        <div class="col">
            <h1>Invite Staff Member</h1>
            <p class="text-muted">Send an invitation to a new team member to join your business</p>
        </div>
    </div>

    @php
        $business = auth()->user()->businesses()->first();
        $subscription = $business ? $business->activeSubscription()->with('plan')->first() : null;
        $maxStaff = $subscription ? $subscription->plan->max_staff : 0;
        $currentStaff = $business ? $business->staff()->where('status', 'active')->count() : 0;
        $canAddMore = $currentStaff < $maxStaff;
    @endphp

    <div class="row">
        <div class="col-md-8">
            @if(!$subscription)
                <div class="alert alert-warning" role="alert">
                    <i class="bi bi-exclamation-triangle"></i>
                    <strong>No Active Subscription</strong>
                    <p class="mb-0">You don't have an active subscription. Please upgrade your plan to add staff members.</p>
                </div>
            @elseif(!$canAddMore)
                <div class="alert alert-danger" role="alert">
                    <i class="bi bi-exclamation-circle"></i>
                    <strong>Staff Limit Reached</strong>
                    <p class="mb-0">You've reached the maximum of {{ $maxStaff }} staff members allowed by your current plan. <a href="#" class="alert-link">Upgrade your plan</a> to add more staff.</p>
                </div>
            @endif

            <div class="card">
                <div class="card-body">
                    @if($canAddMore)
                        <form action="{{ route('business.staff.store') }}" method="POST">
                            @csrf

                            <!-- Email Address -->
                            <div class="mb-4">
                                <label for="email" class="form-label fw-bold">Email Address</label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror"
                                       id="email" name="email" value="{{ old('email') }}"
                                       placeholder="staff@example.com" required>
                                <small class="text-muted">An invitation will be sent to this email address</small>
                                @error('email')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Role Selection -->
                            <div class="mb-4">
                                <label for="role" class="form-label fw-bold">Role</label>
                                <select class="form-select @error('role') is-invalid @enderror"
                                        id="role" name="role" required>
                                    <option value="">-- Select Role --</option>
                                    <option value="manager" {{ old('role') === 'manager' ? 'selected' : '' }}>
                                        Manager (Full Access)
                                    </option>
                                    <option value="staff" {{ old('role') === 'staff' ? 'selected' : '' }}>
                                        Staff Member (Limited Access)
                                    </option>
                                </select>
                                <small class="text-muted">
                                    <strong>Manager:</strong> Can manage venues, offers, staff, and view analytics<br>
                                    <strong>Staff:</strong> Can process redemptions and view limited features
                                </small>
                                @error('role')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Team Assignment -->
                            <div class="mb-4">
                                <label for="team_id" class="form-label fw-bold">Team (Optional)</label>
                                <select class="form-select @error('team_id') is-invalid @enderror"
                                        id="team_id" name="team_id">
                                    <option value="">-- No Specific Team --</option>
                                    @if($business)
                                        @foreach($business->teams as $team)
                                            <option value="{{ $team->id }}" {{ old('team_id') == $team->id ? 'selected' : '' }}>
                                                {{ $team->name }}
                                            </option>
                                        @endforeach
                                    @endif
                                </select>
                                <small class="text-muted">Assign this staff member to a specific team</small>
                                @error('team_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Permissions -->
                            <div class="mb-4">
                                <label class="form-label fw-bold">Permissions</label>
                                <p class="text-muted mb-3">Select which features this staff member can access:</p>

                                <div class="row">
                                    @foreach($importantPermissions as $permission)
                                        <div class="col-md-6 mb-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox"
                                                       id="permission{{ $permission->id }}"
                                                       name="permissions[]"
                                                       value="{{ $permission->id }}"
                                                       {{ in_array($permission->id, old('permissions', [])) ? 'checked' : '' }}>
                                                <label class="form-check-label" for="permission{{ $permission->id }}">
                                                    <strong>{{ $permission->display_name ?? ucwords(str_replace('_', ' ', $permission->name)) }}</strong>
                                                    <br>
                                                    <small class="text-muted">
                                                        @switch($permission->name)
                                                            @case('manage_offers')
                                                                Create, edit, and manage offers
                                                                @break
                                                            @case('manage_venues')
                                                                Create and manage business venues
                                                                @break
                                                            @case('manage_spotlight')
                                                                Manage spotlight features
                                                                @break
                                                            @case('process_redemptions')
                                                                Process customer redemptions
                                                                @break
                                                            @default
                                                                {{ $permission->description ?? '' }}
                                                        @endswitch
                                                    </small>
                                                </label>
                                            </div>
                                        </div>
                                    @endforeach
                                </div>
                                @error('permissions')
                                    <div class="text-danger">{{ $message }}</div>
                                @enderror
                            </div>

                            <!-- Buttons -->
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-paper-plane"></i> Send Invitation
                                </button>
                                <a href="{{ route('business.staff.index') }}" class="btn btn-secondary">
                                    Cancel
                                </a>
                            </div>
                        </form>
                    @else
                        <div class="alert alert-danger text-center py-5">
                            <i class="fas fa-lock" style="font-size: 3rem;"></i>
                            <h4 class="mt-3">Staff Limit Reached</h4>
                            <p class="mb-3">Your current plan allows <strong>{{ $maxStaff }} staff members</strong>.</p>
                            <p class="text-muted mb-4">To add more staff, please upgrade your subscription plan.</p>
                            <a href="{{ route('business.subscription.index') }}" class="btn btn-primary">
                                <i class="fas fa-arrow-up"></i> Upgrade Plan
                            </a>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Info Sidebar -->
        <div class="col-md-4">
            @if($subscription)
                <div class="card mb-3">
                    <div class="card-body">
                        <h5 class="card-title">Plan: {{ $subscription->plan->name }}</h5>
                        <div class="progress mb-3">
                            <div class="progress-bar @if($currentStaff >= $maxStaff) bg-danger @elseif($currentStaff >= $maxStaff * 0.8) bg-warning @else bg-success @endif"
                                 role="progressbar"
                                 style="width: {{ min(100, ($currentStaff / $maxStaff) * 100) }}%"
                                 aria-valuenow="{{ $currentStaff }}"
                                 aria-valuemin="0"
                                 aria-valuemax="{{ $maxStaff }}">
                            </div>
                        </div>
                        <div class="d-flex justify-content-between mb-3">
                            <span><strong>Staff Members:</strong></span>
                            <span class="@if($currentStaff >= $maxStaff) text-danger @endif"><strong>{{ $currentStaff }}/{{ $maxStaff }}</strong></span>
                        </div>
                        @if($canAddMore)
                            <p class="mb-0 text-success"><i class="fas fa-check-circle"></i> <small>You can add {{ $maxStaff - $currentStaff }} more staff member(s)</small></p>
                        @else
                            <p class="mb-0 text-danger"><i class="fas fa-exclamation-circle"></i> <small>Limit reached - upgrade to add more</small></p>
                        @endif
                    </div>
                </div>
            @endif

            <div class="card bg-light">
                <div class="card-body">
                    <h5 class="card-title">How it works</h5>
                    <ol class="small">
                        <li>Enter the staff member's email address</li>
                        <li>Select their role and permissions</li>
                        <li>Click "Send Invitation"</li>
                        <li>They'll receive an email with a link to create their account</li>
                        <li>Once they create their account, they can start working immediately</li>
                    </ol>
                </div>
            </div>

            <div class="card mt-3">
                <div class="card-body">
                    <h5 class="card-title text-warning">Important</h5>
                    <p class="small mb-0">
                        <strong>Invitations expire after 7 days.</strong> If the staff member doesn't accept within this period, you'll need to send a new invitation.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

