@extends('layouts.business')

@section('title', 'Venue Details - ' . $venue->name)

@section('content')
    <div class="mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h2 class="fw-bold mb-1">{{ $venue->name }}</h2>
                <p class="text-muted mb-0">
                    <i class="bi bi-geo-alt me-1"></i>{{ $venue->city }}, {{ $venue->area }}
                </p>
            </div>
            <div class="d-flex gap-2">
                <a href="{{ route('business.venues.edit', $venue->id) }}" class="btn btn-primary">
                    <i class="bi bi-pencil me-2"></i>Edit Venue
                </a>
                <a href="{{ route('business.venues.index') }}" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-2"></i>Back to List
                </a>
            </div>
        </div>
    </div>

    <!-- Status Badge -->
    <div class="mb-4">
        @if($venue->status === 'active')
            <span class="badge bg-success px-3 py-2">
            <i class="bi bi-check-circle me-1"></i>Active
        </span>
        @elseif($venue->status === 'rejected')
            <span class="badge bg-danger px-3 py-2">
            <i class="bi bi-x-circle me-1"></i>Rejected
        </span>
        @elseif($venue->status === 'inactive')
            <span class="badge bg-secondary px-3 py-2">
            <i class="bi bi-x-circle me-1"></i>Inactive
        </span>
        @else
            <span class="badge bg-warning px-3 py-2">
            <i class="bi bi-tools me-1"></i>Under Maintenance
        </span>
        @endif

        <span class="badge bg-primary px-3 py-2 ms-2">
        <i class="bi bi-tag me-1"></i>{{ $venue->category->name }}
    </span>
    </div>

    <!-- Rejection Notice -->
    @if($venue->status === 'rejected' && $venue->rejection_reason)
    <div class="alert alert-danger alert-dismissible fade show mb-4" role="alert">
        <div class="d-flex align-items-start">
            <i class="bi bi-exclamation-circle me-3 mt-1"></i>
            <div>
                <h5 class="alert-heading mb-2">Venue Rejection Notice</h5>
                <p class="mb-2"><strong>Reason:</strong></p>
                <p class="mb-3">{{ $venue->rejection_reason }}</p>

                @php
                    $checklist = $venue->rejection_checklist;
                    if (is_string($checklist)) {
                        $checklist = json_decode($checklist, true);
                    }

                    // Filter only checked items
                    if (is_array($checklist)) {
                        $checklist = array_filter($checklist, function($item) {
                            if (is_array($item)) {
                                return isset($item['checked']) && $item['checked'] === true;
                            }
                            return true;
                        });
                    }
                @endphp
                @if($checklist && is_array($checklist) && count($checklist) > 0)
                    <p><strong>Issues Found:</strong></p>
                    <ul class="mb-0 ps-3">
                        @foreach($checklist as $item)
                            @if(is_array($item) && isset($item['label']))
                                <li>{{ $item['label'] }}</li>
                            @elseif(is_string($item))
                                <li>{{ $item }}</li>
                            @else
                                <li>{{ json_encode($item) }}</li>
                            @endif
                        @endforeach
                    </ul>
                @endif

                <div class="mt-3">
                    <p class="text-muted small mb-2">You can edit the venue and resubmit for approval.</p>
                    <a href="{{ route('business.venues.edit', $venue->id) }}" class="btn btn-sm btn-outline-danger">
                        <i class="bi bi-pencil me-1"></i> Edit & Resubmit
                    </a>
                </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
    </div>
    @endif

    <script>
        // Prevent alert from auto-dismissing
        document.addEventListener('DOMContentLoaded', function() {
            const alerts = document.querySelectorAll('.alert-dismissible');
            alerts.forEach(function(alert) {
                // Remove the auto-dismiss timeout if any
                const bsAlert = new bootstrap.Alert(alert);
                // Remove data-bs-dismiss to prevent dismissal
                const closeBtn = alert.querySelector('.btn-close');
                if (closeBtn) {
                    closeBtn.addEventListener('click', function(e) {
                        e.preventDefault();
                        // Make it manually dismissible but don't auto-dismiss
                        alert.style.display = 'none';
                    });
                }
            });
        });
    </script>

    <!-- Stats Overview -->
    <div class="row g-4 mb-4">
        <div class="col-md-6 col-lg-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-shrink-0">
                            <div class="bg-primary bg-opacity-10 text-primary rounded p-3">
                                <i class="bi bi-ticket-perforated fs-4"></i>
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <div class="text-muted small">Total Offers</div>
                            <div class="fs-4 fw-bold">{{ number_format($stats['total_offers']) }}</div>
                            <small class="text-success">{{ $stats['active_offers'] }} active</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-6 col-lg-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-shrink-0">
                            <div class="bg-success bg-opacity-10 text-success rounded p-3">
                                <i class="bi bi-check-circle fs-4"></i>
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <div class="text-muted small">Redemptions</div>
                            <div class="fs-4 fw-bold">{{ number_format($stats['total_redemptions']) }}</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-6 col-lg-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-shrink-0">
                            <div class="bg-warning bg-opacity-10 text-warning rounded p-3">
                                <i class="bi bi-star fs-4"></i>
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <div class="text-muted small">Reviews</div>
                            <div class="fs-4 fw-bold">{{ number_format($stats['total_reviews']) }}</div>
                            <small class="text-muted">
                                @if($stats['average_rating'])
                                    {{ number_format($stats['average_rating'], 1) }}
                                    <i class="bi bi-star-fill text-warning"></i>
                                @else
                                    No ratings yet
                                @endif
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-6 col-lg-3">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-shrink-0">
                            <div class="bg-info bg-opacity-10 text-info rounded p-3">
                                <i class="bi bi-eye fs-4"></i>
                            </div>
                        </div>
                        <div class="flex-grow-1 ms-3">
                            <div class="text-muted small">Total Views</div>
                            <div class="fs-4 fw-bold">{{ number_format($stats['total_views']) }}</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row g-4">
        <!-- Venue Details -->
        <div class="col-lg-8">
            <!-- Featured Image -->
            @if($venue->featured_image)
                <div class="card border-0 shadow-sm mb-4">
                    <img src="{{ Storage::url($venue->featured_image) }}"
                         class="card-img-top"
                         alt="{{ $venue->name }}"
                         style="height: 400px; object-fit: cover;">
                </div>
            @endif

            <!-- Description -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">About</h5>
                </div>
                <div class="card-body">
                    <p class="mb-0">{{ $venue->description ?? 'No description available.' }}</p>
                </div>
            </div>

            <!-- Gallery -->
            @if($venue->images && count(json_decode($venue->images, true)) > 0)
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-white border-bottom">
                        <h5 class="mb-0 fw-bold">Gallery</h5>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            @foreach(json_decode($venue->images, true) as $image)
                                <div class="col-md-4">
                                    <img src="{{ Storage::url($image) }}"
                                         class="img-fluid rounded"
                                         style="height: 200px; width: 100%; object-fit: cover;">
                                </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            @endif

            <!-- Recent Redemptions -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Recent Redemptions</h5>
                </div>
                <div class="card-body p-0">
                    @if($recentRedemptions->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="bg-light">
                                <tr>
                                    <th>Customer</th>
                                    <th>Offer</th>
                                    <th>Code</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                </tr>
                                </thead>
                                <tbody>
                                @foreach($recentRedemptions as $redemption)
                                    <tr>
                                        <td>{{ $redemption->user->full_name }}</td>
                                        <td>{{ $redemption->offer->title }}</td>
                                        <td>
                                            <code>{{ $redemption->redemption_code }}</code>
                                        </td>
                                        <td>{{ $redemption->created_at->format('M d, Y') }}</td>
                                        <td>
                                            @if($redemption->status === 'redeemed')
                                                <span class="badge bg-success">Redeemed</span>
                                            @elseif($redemption->status === 'pending')
                                                <span class="badge bg-warning">Pending</span>
                                            @else
                                                <span class="badge bg-secondary">{{ ucfirst($redemption->status) }}</span>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <div class="text-center py-5 text-muted">
                            <i class="bi bi-inbox" style="font-size: 3rem;"></i>
                            <p class="mt-3 mb-0">No redemptions yet</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Contact Information -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Contact Information</h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <div class="text-muted small mb-1">Phone</div>
                        <div class="fw-semibold">
                            <i class="bi bi-telephone me-2"></i>{{ $venue->phone }}
                        </div>
                    </div>

                    @if($venue->email)
                        <div class="mb-3">
                            <div class="text-muted small mb-1">Email</div>
                            <div class="fw-semibold">
                                <i class="bi bi-envelope me-2"></i>{{ $venue->email }}
                            </div>
                        </div>
                    @endif

                    @if($venue->website)
                        <div class="mb-3">
                            <div class="text-muted small mb-1">Website</div>
                            <div class="fw-semibold">
                                <i class="bi bi-globe me-2"></i>
                                <a href="{{ $venue->website }}" target="_blank">Visit Website</a>
                            </div>
                        </div>
                    @endif

                    <div class="mb-0">
                        <div class="text-muted small mb-1">Address</div>
                        <div class="fw-semibold">
                            <i class="bi bi-geo-alt me-2"></i>{{ $venue->address }}
                        </div>
                    </div>
                </div>
            </div>

            <!-- Location Map -->
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Location</h5>
                </div>
                <div class="card-body p-0">
                    <div id="map" style="height: 250px; width: 100%;">
                        <div class="d-flex align-items-center justify-content-center h-100 bg-light">
                            <div class="text-center text-muted">
                                <i class="bi bi-map" style="font-size: 2rem;"></i>
                                <p class="mt-2 mb-0 small">Map integration required</p>
                                <p class="text-muted small">
                                    {{ $venue->latitude }}, {{ $venue->longitude }}
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-white border-bottom">
                    <h5 class="mb-0 fw-bold">Quick Actions</h5>
                </div>
                <div class="card-body">
                    <a href="{{ route('business.offers.create', ['venue_id' => $venue->id]) }}"
                       class="btn btn-primary w-100 mb-2">
                        <i class="bi bi-plus-circle me-2"></i>Create Offer
                    </a>
                    <a href="{{ route('business.venues.edit', $venue->id) }}"
                       class="btn btn-outline-primary w-100 mb-2">
                        <i class="bi bi-pencil me-2"></i>Edit Details
                    </a>
                    <button class="btn btn-outline-secondary w-100"
                            onclick="toggleStatus({{ $venue->id }})">
                        <i class="bi bi-power me-2"></i>
                        {{ $venue->status === 'active' ? 'Deactivate' : 'Activate' }}
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
    <script>
        function toggleStatus(venueId) {
            if (confirm('Are you sure you want to change the venue status?')) {
                fetch(`/business/venues/${venueId}/toggle-status`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            location.reload();
                        }
                    })
                    .catch(error => console.error('Error:', error));
            }
        }
    </script>
@endpush
