<?php

namespace App\Http\Controllers\API\V1\Business;

use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class BusinessRegistrationController extends Controller
{
    /**
     * Register a new business
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            // Owner Information
            'owner_first_name' => 'required|string|max:255',
            'owner_last_name' => 'required|string|max:255',
            'owner_email' => 'required|email|unique:users,email',
            'owner_phone' => 'required|string|unique:users,phone',
            'password' => 'required|string|min:8|confirmed',

            // Business Information
            'business_name' => 'required|string|max:255',
            'business_name_ar' => 'nullable|string|max:255',
            'trade_license_number' => 'required|string|unique:businesses,trade_license_number',
            'trade_license_document' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120', // 5MB
            'email' => 'required|email',
            'phone' => 'required|string',
            'website' => 'nullable|url',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        DB::beginTransaction();

        try {
            // Create user account (business owner)
            $user = User::create([
                'first_name' => $validated['owner_first_name'],
                'last_name' => $validated['owner_last_name'],
                'full_name' => $validated['owner_first_name'] . ' ' . $validated['owner_last_name'],
                'email' => $validated['owner_email'],
                'phone' => $validated['owner_phone'],
                'password' => Hash::make($validated['password']),
                'account_type' => 'business_owner',
                'role' => 'business_owner',
                'status' => 'active',
                'email_verified_at' => now(), // Auto-verify for business
            ]);

            // Handle file uploads
            $tradeLicensePath = $request->file('trade_license_document')->store('licenses', 'public');
            $logoPath = $request->hasFile('logo')
                ? $request->file('logo')->store('logos', 'public')
                : null;
            $coverPath = $request->hasFile('cover_image')
                ? $request->file('cover_image')->store('covers', 'public')
                : null;

            // Create business
            $business = Business::create([
                'owner_id' => $user->id,
                'business_name' => $validated['business_name'],
                'business_name_ar' => $validated['business_name_ar'] ?? null,
                'trade_license_number' => $validated['trade_license_number'],
                'trade_license_document' => $tradeLicensePath,
                'email' => $validated['email'],
                'phone' => $validated['phone'],
                'website' => $validated['website'] ?? null,
                'description' => $validated['description'] ?? null,
                'description_ar' => $validated['description_ar'] ?? null,
                'logo' => $logoPath,
                'cover_image' => $coverPath,
                'status' => 'pending', // Pending admin verification
                'subscription_status' => 'inactive',
                'total_venues' => 0,
                'total_staff' => 0,
                'total_offers' => 0,
                'rating' => 0,
                'total_reviews' => 0,
            ]);

            DB::commit();

            // Generate API token
            $token = $user->createToken('business-token')->plainTextToken;

            return response()->json([
                'success' => true,
                'message' => 'Business registered successfully. Your application is pending verification.',
                'data' => [
                    'user' => $user,
                    'business' => $business,
                    'token' => $token,
                ]
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Registration failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get business registration status
     */
    public function status()
    {
        $user = auth()->user();
        $business = $user->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business associated with this account'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'status' => $business->status,
                'verification_notes' => $business->verification_notes,
                'verified_at' => $business->verified_at,
                'subscription_status' => $business->subscription_status,
                'subscription_plan' => $business->subscriptionPlan,
                'subscription_end_date' => $business->subscription_end_date,
            ]
        ]);
    }

    /**
     * Update business profile
     */
    public function updateProfile(Request $request)
    {
        $user = auth()->user();
        $business = $user->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business associated with this account'
            ], 404);
        }

        $validated = $request->validate([
            'business_name' => 'sometimes|string|max:255',
            'business_name_ar' => 'nullable|string|max:255',
            'email' => 'sometimes|email',
            'phone' => 'sometimes|string',
            'website' => 'nullable|url',
            'description' => 'nullable|string',
            'description_ar' => 'nullable|string',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'cover_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            if ($business->logo) {
                Storage::disk('public')->delete($business->logo);
            }
            $validated['logo'] = $request->file('logo')->store('logos', 'public');
        }

        // Handle cover image upload
        if ($request->hasFile('cover_image')) {
            if ($business->cover_image) {
                Storage::disk('public')->delete($business->cover_image);
            }
            $validated['cover_image'] = $request->file('cover_image')->store('covers', 'public');
        }

        $business->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'Business profile updated successfully',
            'data' => $business
        ]);
    }

    /**
     * Get business dashboard statistics
     */
    public function dashboard()
    {
        $business = auth()->user()->business;

        if (!$business) {
            return response()->json([
                'success' => false,
                'message' => 'No business associated with this account'
            ], 404);
        }

        $stats = [
            'business' => $business,
            'total_venues' => $business->total_venues,
            'total_offers' => $business->total_offers,
            'active_offers' => $business->offers()->where('status', 'active')->count(),
            'pending_offers' => $business->offers()->where('approval_status', 'pending')->count(),
            'total_redemptions' => $business->offers()->sum('redemption_count'),
            'total_views' => $business->offers()->sum('view_count'),
            'average_rating' => $business->rating,
            'total_reviews' => $business->total_reviews,
            'staff_count' => $business->staff()->count(),
            'subscription' => [
                'status' => $business->subscription_status,
                'plan' => $business->subscriptionPlan,
                'start_date' => $business->subscription_start_date,
                'end_date' => $business->subscription_end_date,
                'days_remaining' => $business->subscription_end_date
                    ? now()->diffInDays($business->subscription_end_date, false)
                    : null,
            ],
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    /**
     * Check business name availability
     */
    public function checkAvailability(Request $request)
    {
        $validated = $request->validate([
            'business_name' => 'required|string|max:255',
            'trade_license_number' => 'required|string',
        ]);

        $businessExists = Business::where('business_name', $validated['business_name'])
            ->exists();

        $licenseExists = Business::where('trade_license_number', $validated['trade_license_number'])
            ->exists();

        return response()->json([
            'success' => true,
            'data' => [
                'business_name_available' => !$businessExists,
                'license_number_available' => !$licenseExists,
            ]
        ]);
    }
}
