<?php

namespace App\Http\Controllers\API\V1\FAQ;

use App\Http\Controllers\Controller;
use App\Models\Faq;
use Illuminate\Http\Request;

class FaqController extends Controller
{
    /**
     * Get all FAQs
     */
    public function index(Request $request)
    {
        $query = Faq::where('is_active', true)->orderBy('order');

        if ($request->search) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('question', 'LIKE', "%{$search}%")
                    ->orWhere('answer', 'LIKE', "%{$search}%")
                    ->orWhere('question_ar', 'LIKE', "%{$search}%")
                    ->orWhere('answer_ar', 'LIKE', "%{$search}%");
            });
        }

        $faqs = $query->get()->groupBy('category');

        return response()->json([
            'success' => true,
            'data' => [
                'categories' => Faq::categories(),
                'faqs' => $faqs,
            ],
        ]);
    }

    /**
     * Get FAQs by category
     */
    public function byCategory($category)
    {
        $faqs = Faq::byCategory($category);

        return response()->json([
            'success' => true,
            'data' => $faqs,
        ]);
    }

    /**
     * Get single FAQ
     */
    public function show($id)
    {
        $faq = Faq::where('is_active', true)->findOrFail($id);

        $faq->incrementViews();

        return response()->json([
            'success' => true,
            'data' => $faq,
        ]);
    }

    /**
     * Mark FAQ as helpful
     */
    public function helpful($id)
    {
        $faq = Faq::findOrFail($id);
        $faq->markAsHelpful();

        return response()->json([
            'success' => true,
            'message' => 'Thank you for your feedback',
        ]);
    }

    /**
     * Search FAQs
     */
    public function search(Request $request)
    {
        $validated = $request->validate([
            'query' => 'required|string|min:2',
        ]);

        $faqs = Faq::search($validated['query']);

        return response()->json([
            'success' => true,
            'data' => $faqs,
        ]);
    }
}
