<?php

namespace App\Http\Controllers\API\V1\Venue;

use App\Http\Controllers\Controller;
use App\Models\Venue;
use App\Models\Favorite;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class VenueController extends Controller
{
    public function index(Request $request)
    {
        $query = Venue::with(['business', 'category'])
            ->where('status', 'active');

        // Search by name
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Filter by category
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by city
        if ($request->has('city')) {
            $query->where('city', $request->city);
        }

        // Filter by area
        if ($request->has('area')) {
            $query->where('area', $request->area);
        }

        // Filter by rating
        if ($request->has('min_rating')) {
            $query->where('rating', '>=', $request->min_rating);
        }

        // Featured venues
        if ($request->has('featured') && $request->featured) {
            $query->where('is_featured', true);
        }

        // Near me (location-based search)
        if ($request->has('latitude') && $request->has('longitude')) {
            $lat = $request->latitude;
            $lng = $request->longitude;
            $radius = $request->radius ?? 10; // km

            $query->select('*')
                ->selectRaw(
                    '( 6371 * acos( cos( radians(?) ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(?) ) + sin( radians(?) ) * sin( radians( latitude ) ) ) ) AS distance',
                    [$lat, $lng, $lat]
                )
                ->having('distance', '<=', $radius)
                ->orderBy('distance');
        } else {
            // Default sorting
            $sortBy = $request->sort_by ?? 'created_at';
            $sortOrder = $request->sort_order ?? 'desc';

            if (in_array($sortBy, ['rating', 'view_count', 'total_reviews', 'created_at'])) {
                $query->orderBy($sortBy, $sortOrder);
            }
        }

        $perPage = $request->per_page ?? 15;
        $venues = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $venues
        ]);
    }

    public function show($id)
    {
        $venue = Venue::with([
            'business',
            'category',
            'reviews' => function($query) {
                $query->where('status', 'approved')
                    ->orderBy('created_at', 'desc')
                    ->take(10);
            },
            'offers' => function($query) {
                $query->where('status', 'active')
                    ->where('start_date', '<=', now())
                    ->where('end_date', '>=', now());
            }
        ])->findOrFail($id);

        // Increment view count
        $venue->increment('view_count');

        // Check if user has favorited
        $isFavorited = false;
        if (auth('sanctum')->check()) {
            $isFavorited = Favorite::where('user_id', auth()->id())
                ->where('favorable_type', Venue::class)
                ->where('favorable_id', $venue->id)
                ->exists();
        }

        return response()->json([
            'success' => true,
            'data' => [
                'venue' => $venue,
                'is_favorited' => $isFavorited,
            ]
        ]);
    }

    public function toggleFavorite($id)
    {
        $venue = Venue::findOrFail($id);

        $favorite = Favorite::where('user_id', auth()->id())
            ->where('favorable_type', Venue::class)
            ->where('favorable_id', $venue->id)
            ->first();

        if ($favorite) {
            $favorite->delete();
            $message = 'Venue removed from favorites';
            $isFavorited = false;
        } else {
            Favorite::create([
                'user_id' => auth()->id(),
                'favorable_type' => Venue::class,
                'favorable_id' => $venue->id,
            ]);
            $message = 'Venue added to favorites';
            $isFavorited = true;
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => [
                'is_favorited' => $isFavorited
            ]
        ]);
    }

    public function favorites(Request $request)
    {
        $favorites = Favorite::with('favorable')
            ->where('user_id', auth()->id())
            ->where('favorable_type', Venue::class)
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $favorites
        ]);
    }

    public function nearby(Request $request)
    {
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'radius' => 'nullable|numeric|min:1|max:50',
        ]);

        $lat = $request->latitude;
        $lng = $request->longitude;
        $radius = $request->radius ?? 10;

        $venues = Venue::select('*')
            ->selectRaw(
                '( 6371 * acos( cos( radians(?) ) * cos( radians( latitude ) ) * cos( radians( longitude ) - radians(?) ) + sin( radians(?) ) * sin( radians( latitude ) ) ) ) AS distance',
                [$lat, $lng, $lat]
            )
            ->where('status', 'active')
            ->having('distance', '<=', $radius)
            ->orderBy('distance')
            ->with(['category', 'business'])
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $venues
        ]);
    }

    public function cities()
    {
        $cities = Venue::select('city')
            ->where('status', 'active')
            ->distinct()
            ->orderBy('city')
            ->pluck('city');

        return response()->json([
            'success' => true,
            'data' => $cities
        ]);
    }

    public function areas(Request $request)
    {
        $query = Venue::select('area')
            ->where('status', 'active')
            ->distinct();

        if ($request->has('city')) {
            $query->where('city', $request->city);
        }

        $areas = $query->orderBy('area')->pluck('area');

        return response()->json([
            'success' => true,
            'data' => $areas
        ]);
    }
}
