<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;

use Illuminate\Support\Str;

class CategoryController extends Controller
{
    /**
     * Display category listing
     */
    public function index()
    {
        $categories = Category::withCount(['businesses', 'offers'])
            ->orderBy('order')
            ->get();

        $stats = [
            'total' => Category::count(),
            'active' => Category::where('is_active', true)->count(),
            'businesses' => Category::withCount('businesses')->get()->sum('businesses_count'),
            'offers' => Category::withCount('offers')->get()->sum('offers_count'),
        ];

        return view('admin.categories.index', compact('categories', 'stats'));
    }

    /**
     * Store new category
     */
    public function store(Request $request)
    {
//        getting is_active on off issue ut i need boolean
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'icon' => 'nullable|string|max:50',
            'sort_order' => 'nullable|integer',
        ]);
//        slug
        $validated['is_active'] = $request->has('is_active') ? (bool)$request->input('is_active') : false;
        $validated['slug'] = \Str::slug($validated['name']);
        $category = Category::create($validated);

        return redirect()->back()->with('success', 'Category created successfully');
    }

    /**
     * Update category
     */
    public function update(Request $request, $id)
    {
        $category = Category::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'icon' => 'nullable|string|max:50',
            'sort_order' => 'nullable|integer',
        ]);

        $category->update($validated);

        return redirect()->back()->with('success', 'Category updated successfully');
    }

    /**
     * Toggle category status
     */
    public function toggle($id)
    {
        $category = Category::findOrFail($id);
        $category->update(['is_active' => !$category->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'Category status updated',
            'data' => $category
        ]);
    }

    /**
     * Delete category
     */
    public function destroy($id)
    {
        $category = Category::findOrFail($id);

        // Check if category has businesses or offers
        if ($category->businesses()->exists() || $category->offers()->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete category with existing businesses or offers'
            ], 400);
        }

        $category->delete();

        return response()->json([
            'success' => true,
            'message' => 'Category deleted successfully'
        ]);
    }

    public function show($id)
    {
        $category = Category::findOrFail($id);
        return response()->json($category);
    }

}
