<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SystemConfiguration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class ConfigurationController extends Controller
{
    /**
     * Display system configurations
     */
    public function index()
    {
        $configurations = SystemConfiguration::orderBy('group')
            ->orderBy('sort_order')
            ->get()
            ->groupBy('group');

        $groups = [
            'general' => 'General Settings',
            'authentication' => 'Authentication & Security',
            'referral' => 'Referral System',
            'spotlight' => 'Spotlight Settings',
            'venue' => 'Venue Management',
            'offer' => 'Offer Management',
            'notification' => 'Notifications',
            'sms' => 'SMS Provider',
            'chat' => 'Chat & Messaging',
            'gdpr' => 'GDPR Compliance',
        ];

        return view('admin.configurations.index', compact('configurations', 'groups'));
    }

    /**
     * Update configuration value
     */
    public function update(Request $request)
    {
        $validated = $request->validate([
            'key' => 'required|exists:system_configurations,key',
            'value' => 'nullable',
        ]);

        $config = SystemConfiguration::where('key', $validated['key'])->first();

        if (!$config) {
            return response()->json([
                'success' => false,
                'message' => 'Configuration not found',
            ], 404);
        }

        // Handle different types
        $value = $validated['value'];

        if ($config->type === 'boolean') {
            $value = filter_var($value, FILTER_VALIDATE_BOOLEAN) ? 'true' : 'false';
        } elseif ($config->type === 'json' || $config->type === 'array') {
            $value = is_array($value) ? json_encode($value) : $value;
        }

        $config->value = $value;
        $config->save();

        // Clear cache
        Cache::forget("config.{$validated['key']}");
        Cache::forget("config.group.{$config->group}");
        Cache::forget('config.public');

        return response()->json([
            'success' => true,
            'message' => 'Configuration updated successfully',
            'data' => $config,
        ]);
    }

    /**
     * Update multiple configurations
     */
    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'configurations' => 'required|array',
            'configurations.*.key' => 'required|exists:system_configurations,key',
            'configurations.*.value' => 'nullable',
        ]);

        $updatedCount = 0;

        foreach ($validated['configurations'] as $configData) {
            $config = SystemConfiguration::where('key', $configData['key'])->first();

            if (!$config) {
                continue;
            }

            $value = $configData['value'];

            if ($config->type === 'boolean') {
                $value = filter_var($value, FILTER_VALIDATE_BOOLEAN) ? 'true' : 'false';
            } elseif ($config->type === 'json' || $config->type === 'array') {
                $value = is_array($value) ? json_encode($value) : $value;
            }

            $config->value = $value;
            $config->save();

            Cache::forget("config.{$configData['key']}");
            $updatedCount++;
        }

        // Clear group and public caches
        SystemConfiguration::clearCache();

        return response()->json([
            'success' => true,
            'message' => "{$updatedCount} configurations updated successfully",
        ]);
    }

    /**
     * Get configuration by key (API)
     */
    public function show($key)
    {
        $value = SystemConfiguration::get($key);

        if ($value === null) {
            return response()->json([
                'success' => false,
                'message' => 'Configuration not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'key' => $key,
                'value' => $value,
            ],
        ]);
    }

    /**
     * Get public configurations (for mobile app)
     */
    public function getPublic()
    {
        $publicConfigs = SystemConfiguration::getPublic();

        return response()->json([
            'success' => true,
            'data' => $publicConfigs,
        ]);
    }

    /**
     * Clear configuration cache
     */
    public function clearCache()
    {
        SystemConfiguration::clearCache();

        return response()->json([
            'success' => true,
            'message' => 'Configuration cache cleared successfully',
        ]);
    }

    /**
     * Reset to default values
     */
    public function resetDefaults(Request $request)
    {
        $validated = $request->validate([
            'group' => 'nullable|string',
        ]);

        $query = SystemConfiguration::query();

        if (isset($validated['group'])) {
            $query->where('group', $validated['group']);
        }

        // This would require storing default values
        // For now, just clear the cache
        SystemConfiguration::clearCache();

        return response()->json([
            'success' => true,
            'message' => 'Configurations reset to defaults',
        ]);
    }

    /**
     * Export configurations
     */
    public function export()
    {
        $configurations = SystemConfiguration::all();

        $exportData = $configurations->map(function ($config) {
            return [
                'key' => $config->key,
                'group' => $config->group,
                'value' => $config->value,
                'type' => $config->type,
                'description' => $config->description,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $exportData,
        ]);
    }

    /**
     * Import configurations
     */
    public function import(Request $request)
    {
        $validated = $request->validate([
            'configurations' => 'required|array',
            'configurations.*.key' => 'required',
            'configurations.*.value' => 'nullable',
        ]);

        $importedCount = 0;

        foreach ($validated['configurations'] as $configData) {
            $config = SystemConfiguration::where('key', $configData['key'])->first();

            if ($config) {
                $config->value = $configData['value'];
                $config->save();
                $importedCount++;
            }
        }

        SystemConfiguration::clearCache();

        return response()->json([
            'success' => true,
            'message' => "{$importedCount} configurations imported successfully",
        ]);
    }
}
