<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\GdprRequest;
use App\Models\User;
use App\Services\GDPR\DataDeletionService;
use App\Services\GDPR\DataExportService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class GdprController extends Controller
{
    protected $exportService;
    protected $deletionService;

    public function __construct(DataExportService $exportService, DataDeletionService $deletionService)
    {
        $this->exportService = $exportService;
        $this->deletionService = $deletionService;
    }

    /**
     * List all GDPR requests
     */
    public function index(Request $request)
    {
        $query = GdprRequest::with('user')->latest();

        if ($request->type) {
            $query->where('type', $request->type);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        $requests = $query->paginate(20);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $requests,
            ]);
        }

        return view('admin.gdpr.index', compact('requests'));
    }

    /**
     * Show single GDPR request
     */
    public function show($id)
    {
        $gdprRequest = GdprRequest::with('user')->findOrFail($id);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'data' => $gdprRequest,
            ]);
        }

        return view('admin.gdpr.show', compact('gdprRequest'));
    }

    /**
     * Process GDPR request
     */
    public function process(Request $request, $id)
    {
        $gdprRequest = GdprRequest::findOrFail($id);

        if ($gdprRequest->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'This request has already been processed',
            ], 400);
        }

        $gdprRequest->update([
            'status' => 'processing',
            'processed_by' => auth()->id(),
            'processed_at' => now(),
        ]);

        AuditLog::logAction('gdpr.processing', $gdprRequest);

        return response()->json([
            'success' => true,
            'message' => 'GDPR request is now being processed',
        ]);
    }

    /**
     * Generate data export
     */
    public function generateExport($id)
    {
        $gdprRequest = GdprRequest::findOrFail($id);

        if ($gdprRequest->type !== 'data_export' && $gdprRequest->type !== 'data_access') {
            return response()->json([
                'success' => false,
                'message' => 'This request is not for data export',
            ], 400);
        }

        try {
            $filePath = $this->exportService->exportUserData($gdprRequest->user);

            $gdprRequest->update([
                'status' => 'completed',
                'file_path' => $filePath,
                'completed_at' => now(),
            ]);

            // Notify user
            $gdprRequest->user->notify(new \App\Notifications\GdprRequestProcessedNotification($gdprRequest));

            AuditLog::logAction('gdpr.export_generated', $gdprRequest);

            return response()->json([
                'success' => true,
                'message' => 'Data export generated successfully',
                'data' => ['file_path' => $filePath],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate export: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Complete access request
     */
    public function completeAccessRequest($id)
    {
        $gdprRequest = GdprRequest::findOrFail($id);

        if ($gdprRequest->type !== 'data_access') {
            return response()->json([
                'success' => false,
                'message' => 'This is not a data access request',
            ], 400);
        }

        $gdprRequest->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);

        $gdprRequest->user->notify(new \App\Notifications\GdprRequestProcessedNotification($gdprRequest));

        AuditLog::logAction('gdpr.access_completed', $gdprRequest);

        return response()->json([
            'success' => true,
            'message' => 'Access request marked as completed',
        ]);
    }

    /**
     * Complete rectification
     */
    public function completeRectification(Request $request, $id)
    {
        $validated = $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        $gdprRequest = GdprRequest::findOrFail($id);

        if ($gdprRequest->type !== 'data_rectification') {
            return response()->json([
                'success' => false,
                'message' => 'This is not a rectification request',
            ], 400);
        }

        $gdprRequest->update([
            'status' => 'completed',
            'completed_at' => now(),
            'admin_notes' => $validated['notes'] ?? null,
        ]);

        $gdprRequest->user->notify(new \App\Notifications\GdprRequestProcessedNotification($gdprRequest));

        AuditLog::logAction('gdpr.rectification_completed', $gdprRequest);

        return response()->json([
            'success' => true,
            'message' => 'Rectification completed',
        ]);
    }

    /**
     * Delete user data
     */
    public function deleteUserData(Request $request, $id)
    {
        $validated = $request->validate([
            'confirmation' => 'required|boolean|accepted',
        ]);

        $gdprRequest = GdprRequest::findOrFail($id);

        if ($gdprRequest->type !== 'data_deletion') {
            return response()->json([
                'success' => false,
                'message' => 'This is not a deletion request',
            ], 400);
        }

        try {
            $result = $this->deletionService->deleteUserData($gdprRequest->user);

            $gdprRequest->update([
                'status' => 'completed',
                'completed_at' => now(),
            ]);

            AuditLog::logAction('gdpr.deletion_completed', null, null, null, [
                'user_id' => $gdprRequest->user_id,
                'records_deleted' => $result,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'User data deleted successfully',
                'data' => $result,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete data: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Reject GDPR request
     */
    public function reject(Request $request, $id)
    {
        $validated = $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        $gdprRequest = GdprRequest::findOrFail($id);

        $gdprRequest->update([
            'status' => 'rejected',
            'processed_by' => auth()->id(),
            'processed_at' => now(),
            'admin_notes' => $validated['reason'],
        ]);

        $gdprRequest->user->notify(new \App\Notifications\GdprRequestProcessedNotification($gdprRequest));

        AuditLog::logAction('gdpr.rejected', $gdprRequest);

        return response()->json([
            'success' => true,
            'message' => 'GDPR request rejected',
        ]);
    }
}
