<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Permission;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PermissionController extends Controller
{
    public function index(Request $request)
    {
        $group = $request->get('group', 'all');

        $query = Permission::with('roles');

        if ($group !== 'all') {
            $query->where('group', $group);
        }

        $permissions = $query->orderBy('group')->orderBy('name')->paginate(50);

        $groups = Permission::allGroups();
        $stats = [
            'total' => Permission::count(),
            'groups' => count($groups),
        ];

        return view('admin.permissions.index', compact('permissions', 'groups', 'stats', 'group'));
    }

    public function create()
    {
        $groups = Permission::allGroups();

        return view('admin.permissions.create', compact('groups'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:permissions,name',
            'display_name' => 'required|string|max:255',
            'group' => 'required|string|max:100',
            'description' => 'nullable|string',
        ]);

        Permission::create([
            'name' => Str::slug($validated['name'], '_'),
            'display_name' => $validated['display_name'],
            'group' => $validated['group'],
            'description' => $validated['description'] ?? null,
        ]);

        return redirect()
            ->route('admin.permissions.index')
            ->with('success', 'Permission created successfully');
    }

    public function show($id)
    {
        $permission = Permission::with('roles')->findOrFail($id);

        $roleCount = $permission->roles()->count();

        return view('admin.permissions.show', compact('permission', 'roleCount'));
    }

    public function edit($id)
    {
        $permission = Permission::findOrFail($id);
        $groups = Permission::allGroups();

        return view('admin.permissions.edit', compact('permission', 'groups'));
    }

    public function update(Request $request, $id)
    {
        $permission = Permission::findOrFail($id);

        $validated = $request->validate([
            'display_name' => 'required|string|max:255',
            'group' => 'required|string|max:100',
            'description' => 'nullable|string',
        ]);

        $permission->update($validated);

        return redirect()
            ->route('admin.permissions.index')
            ->with('success', 'Permission updated successfully');
    }

    public function destroy($id)
    {
        $permission = Permission::findOrFail($id);

        $roleCount = $permission->roles()->count();

        if ($roleCount > 0) {
            return response()->json([
                'success' => false,
                'message' => "Cannot delete permission. {$roleCount} roles have this permission."
            ], 400);
        }

        $permission->delete();

        return response()->json([
            'success' => true,
            'message' => 'Permission deleted successfully'
        ]);
    }

    public function roles($id)
    {
        $permission = Permission::with('roles')->findOrFail($id);

        return view('admin.permissions.roles', compact('permission'));
    }

    public function bulkAssign(Request $request)
    {
        $validated = $request->validate([
            'role_id' => 'required|exists:roles,id',
            'permissions' => 'required|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role = Role::findOrFail($validated['role_id']);

        if ($role->is_system) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot modify system role permissions'
            ], 403);
        }

        $role->permissions()->syncWithoutDetaching($validated['permissions']);

        return response()->json([
            'success' => true,
            'message' => 'Permissions assigned successfully'
        ]);
    }
}
